/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */package org.jbpm.util;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectStreamClass;

import org.jbpm.graph.exe.ProcessInstance;

/**
 * Implementation of an object input stream that delegates class loading to the
 * thread context classloader in preference.
 * 
 * @author kevin
 */
public class ContextObjectInputStream extends ObjectInputStream {
	/**
	 * Construct a default instance of the context aware object input stream.
	 * 
	 * @throws SecurityException
	 *           if an active security manager prevents subclassing
	 * @throws IOException
	 *           for IO errors.
	 */
	protected ContextObjectInputStream() throws IOException, SecurityException {}

	/**
	 * Construct a context aware object input stream using the specified input
	 * stream.
	 * 
	 * @param in
	 *          The associated input stream.
	 * @throws IOException
	 *           for IO errors reading from the stream.
	 */
	public ContextObjectInputStream(final InputStream in) throws IOException {
		super(in);
	}

	/**
	 * Construct a context aware object input stream using the specified input
	 * stream.
	 * 
	 * @param in
	 *          The associated input stream.
	 * @param processInstance
	 *          The associated process instance.
	 * @throws IOException
	 *           for IO errors reading from the stream.
	 */
	public ContextObjectInputStream(final InputStream in,
			ProcessInstance processInstance) throws IOException {
		super(in);
		this.setProcessInstance(processInstance);
		ClassUtil.setProcessInstance(processInstance);
	}

	/**
	 * Resolve the class using the active context.
	 * 
	 * @param desc
	 *          The object stream class description.
	 * @return The associated class.
	 * @throws ClassNotFoundException
	 *           if the class cannot be found.
	 * @throws IOException
	 *           for I/O Errors.
	 */
	protected Class resolveClass(final ObjectStreamClass desc)
			throws IOException, ClassNotFoundException {
		return ClassUtil.forName(desc.getName(), ContextObjectInputStream.class);
	}

	/**
	 * Resolve the proxy class using the active context.
	 * 
	 * @param interfaces
	 *          The interfaces associated with the proxy.
	 * @return The associated class.
	 * @throws ClassNotFoundException
	 *           if an interface cannot be found.
	 * @throws IOException
	 *           for I/O Errors.
	 */
	protected Class resolveProxyClass(String[] interfaces) throws IOException,
			ClassNotFoundException {
		return ClassUtil.resolveProxy(interfaces, ContextObjectInputStream.class);
	}

	public void setProcessInstance(ProcessInstance processInstance) {
		ClassUtil.setProcessInstance(processInstance);
	}

	public ProcessInstance getProcessInstance() {
		return this.processInstance;
	}

	ProcessInstance processInstance;
}
