/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.gd.jpdl.ui.editor;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.FileEditorInput;
import org.jbpm.gd.common.notation.BendPoint;
import org.jbpm.gd.common.notation.Edge;
import org.jbpm.gd.common.notation.Node;
import org.jbpm.gd.common.notation.NodeContainer;
import org.jbpm.gd.common.notation.NotationElement;
import org.jbpm.gd.common.notation.NotationElementContainer;
import org.jbpm.gd.common.notation.RootContainer;
import org.jbpm.gd.jpdl.model.JpdlElement;
import org.jbpm.gd.jpdl.model.NamedElement;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.ProcessDefinition;
import org.jbpm.gd.jpdl.model.Transition;
import org.jbpm.gd.jpdl.ui.JpdlLogger;
import org.jbpm.gd.jpdl.ui.JpdlPlugin;
import org.jbpm.gd.jpdl.ui.notation.JpdlNotationMapping;

public class JpdlContentProvider {

	public static final JpdlContentProvider INSTANCE = new JpdlContentProvider();

	
	// Writing the stuff
	
	public void saveToInput(
			IEditorInput input,
			RootContainer processDefinitionNotationElement) {
		try {
			getGpdFile(((IFileEditorInput)input).getFile()).setContents(
					new ByteArrayInputStream(toGraphicsXml(processDefinitionNotationElement).getBytes()), true, true, null);
		} catch (Exception e) {
			e.printStackTrace();
		}
	 }	
		
	private String toGraphicsXml(RootContainer processDefinitionNotationElement) {
		StringWriter writer = new StringWriter();
		write(processDefinitionNotationElement, writer);
		return writer.toString();
	}

	private void write(
			RootContainer processDefinitionNotationElement, Writer writer) {
		try {
			Document document = DocumentHelper.createDocument();
			Element root = document.addElement("root-container");
			write(processDefinitionNotationElement, root);
			XMLWriter xmlWriter = new XMLWriter(writer, OutputFormat.createPrettyPrint());
			xmlWriter.write(document);
		} catch (IOException e) {
			e.printStackTrace(new PrintWriter(writer));
		}
	}
	
	private void write(
			RootContainer processDefinitionNotationElement,
			Element element) {
		addAttribute(element, "name", ((NamedElement)processDefinitionNotationElement.getSemanticElement()).getName());
		addAttribute(element, "width", Integer.toString(processDefinitionNotationElement.getDimension().width));
		addAttribute(element, "height", Integer.toString(processDefinitionNotationElement.getDimension().height));
		Iterator iter = processDefinitionNotationElement.getNodes().iterator();
		while (iter.hasNext()) {
			write((Node) iter.next(), element);
		}
	}
	
	private void write(Node node, Element element) {
		Element newElement = null;
		if (node instanceof NodeContainer) {
			newElement = addElement(element, "node-container");
		} else {
			newElement = addElement(element, "node");
		}
		addAttribute(newElement, "name", ((NamedElement)node.getSemanticElement()).getName());
		addAttribute(newElement, "x", String.valueOf(node.getConstraint().x));
		addAttribute(newElement, "y", String.valueOf(node.getConstraint().y));
		addAttribute(newElement, "width", String.valueOf(node.getConstraint().width));
		addAttribute(newElement, "height", String.valueOf(node.getConstraint().height));
		if (node instanceof NodeContainer) {
			Iterator nodes = ((NodeContainer)node).getNodes().iterator();
			while (nodes.hasNext()) {
				write((Node)nodes.next(), newElement); 
			}
		}
		Iterator edges = node.getLeavingEdges().iterator();
		while (edges.hasNext()) {
			Edge edge = (Edge) edges.next();
			write(edge, addElement(newElement, "edge"));
		}
	}

	private void write(Edge edge,
			Element element) {
		Point offset = edge.getLabel().getOffset();
		if (offset != null) {
			Element label = addElement(element, "label");
			addAttribute(label, "x", String.valueOf(offset.x));
			addAttribute(label, "y", String.valueOf(offset.y));
		}
		Iterator bendpoints = edge.getBendPoints().iterator();
		while (bendpoints.hasNext()) {
			write((BendPoint) bendpoints.next(), addElement(element, "bendpoint"));
		}
	}

	private void write(BendPoint bendpoint, Element bendpointElement) {
		addAttribute(bendpointElement, "w1", String.valueOf(bendpoint
				.getFirstRelativeDimension().width));
		addAttribute(bendpointElement, "h1", String.valueOf(bendpoint
				.getFirstRelativeDimension().height));
		addAttribute(bendpointElement, "w2", String.valueOf(bendpoint
				.getSecondRelativeDimension().width));
		addAttribute(bendpointElement, "h2", String.valueOf(bendpoint
				.getSecondRelativeDimension().height));
	}
	
	private Element addElement(Element element, String elementName) {
		Element newElement = element.addElement(elementName);
		return newElement;
	}

	private void addAttribute(Element e, String attributeName,
			String value) {
		if (value != null) {
			e.addAttribute(attributeName, value);
		}
	}

	// Reading the stuff

	public FileEditorInput getJpdlEditorInput(IEditorInput input) {
		return new FileEditorInput(getJpdlFile(((IFileEditorInput)input).getFile()));
	}
	
	public FileEditorInput getGpdEditorInput(IEditorInput input) {
		return new FileEditorInput(getGpdFile(((IFileEditorInput)input).getFile()));
	}
	
	private IFile getJpdlFile(IFile gpdFile) {
		IProject project = gpdFile.getProject();
		IPath gpdPath = gpdFile.getProjectRelativePath();
		IPath jpdlPath = gpdPath.removeLastSegments(1).append(getJpdlFileName(gpdFile));
		IFile jpdlFile = project.getFile(jpdlPath);
		if (!jpdlFile.exists()) {
			createJpdlFile(jpdlFile);
		}
		return jpdlFile;
	}

	private void createJpdlFile(IFile jpdlFile) {
		try {
			jpdlFile.create(initialProcessDefinitionInfo(), true, null);
		} catch (CoreException e) {
			JpdlLogger.logError(e);
		}
	}
	
	private void createGpdFile(IFile gpdFile) {
		try {
			gpdFile.create(initialGraphicalInfo(), true, null);
		} catch (CoreException e) {
			JpdlLogger.logError(e);
		}
	}
	
	private String getJpdlFileName(IFile gpdFile) {
		String name = gpdFile.getName();
		if ("gpd.xml".equals(name)) {
			return "processdefinition.xml";
		} else {
			return name.substring(5);
		}
	}

	private IFile getGpdFile(IFile jpdlFile) {
		IProject project = jpdlFile.getProject();
		IPath jpdlPath = jpdlFile.getProjectRelativePath();
		IPath gpdPath = jpdlPath.removeLastSegments(1).append(getGpdFileName(jpdlFile));
		IFile gpdFile = project.getFile(gpdPath);
		if (!gpdFile.exists()) {
			createGpdFile(gpdFile);
		}
		return gpdFile;
	}
	
	private String getGpdFileName(IFile jpdlFile) {
		String name = jpdlFile.getName();
		if ("processdefinition.xml".equals(name)) {
			return "gpd.xml";
		} else {
			return ".gpd." + name;
		}
	}

	public void addGraphicalInfo(RootContainer processDefinitionNotationElement,
			IEditorInput input) throws PartInitException {
		try {
			IFile file = ((IFileEditorInput)input).getFile();
			if (file.exists()) {
				addGraphicalInfo(processDefinitionNotationElement,
					new InputStreamReader(((IFileEditorInput)input).getFile().getContents()));
			} else {
				file.create(initialGraphicalInfo(), true, null);
			}
		} catch (CoreException e) {
			e.printStackTrace();
			throw new PartInitException(
					new Status(IStatus.ERROR, JpdlPlugin.getDefault()
							.getClass().getName(), 0,
							"Error reading contents of file : "
									+ input.getName() + ".", null));
		}
	}

	private ByteArrayInputStream initialGraphicalInfo() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		buffer.append("\n\n<root-container />");
		return new ByteArrayInputStream(buffer.toString().getBytes());
	}

	private ByteArrayInputStream initialProcessDefinitionInfo() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		buffer.append("\n\n<process-definition />");
		return new ByteArrayInputStream(buffer.toString().getBytes());
	}

	private void addGraphicalInfo(
			RootContainer processDefinitionNotationElement, Reader reader) {
		try {
			Element processDiagramInfo = new SAXReader().read(reader)
					.getRootElement();
			addGraphicalInfo(processDefinitionNotationElement, processDiagramInfo);
		} catch (DocumentException e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}
	
	private void addNodeGraphicalInfo(NotationElementContainer container, NodeElementContainer nodeElementContainer, List nodeList) {
		for (int i = 0; i < nodeList.size(); i++) {
			Element nodeInfo = (Element)nodeList.get(i);
			NodeElement node = nodeElementContainer.getNodeElementByName(nodeInfo.attributeValue("name"));
			if (node != null) {
				String notationElementId = JpdlNotationMapping.getNotationElementId(node.getElementId());
				Node notationElement = (Node)container.getFactory().create(notationElementId);
				addNodeGraphicalInfo(container, nodeInfo, node, notationElement);
			}
		}
	}

	private void addNodeGraphicalInfo(NotationElementContainer container, Element nodeInfo, NodeElement node, Node notationElement) {
		notationElement.setSemanticElement(node);
		notationElement.register();
		container.addNode(notationElement);
		node.addPropertyChangeListener(notationElement);
		addGraphicalInfo(notationElement, nodeInfo);
	}
	
	private void addContainerGraphicalInfo(NotationElementContainer container, NodeElementContainer nodeElementContainer, List nodeList) {
		for (int i = 0; i < nodeList.size(); i++) {
			Element nodeInfo = (Element)nodeList.get(i);
			NodeElement node = nodeElementContainer.getNodeElementByName(nodeInfo.attributeValue("name"));
			if (node != null) {
				String notationElementId = JpdlNotationMapping.getNotationElementId(node.getElementId());
				Node notationElement = (Node)container.getFactory().create(notationElementId);
				addNodeGraphicalInfo(container, nodeInfo, node, notationElement);
				addNodeGraphicalInfo((NotationElementContainer)notationElement, (NodeElementContainer)node, nodeInfo.elements("node"));
				addContainerGraphicalInfo((NotationElementContainer)notationElement, (NodeElementContainer)node, nodeInfo.elements("node-container"));
			}
		}
	}
	
	private void addGraphicalInfo(
			RootContainer processDefinitionNotationElement,
			Element processDiagramInfo) {
		addProcessDiagramDimension(processDefinitionNotationElement, processDiagramInfo);
		NodeElementContainer nodeElementContainer = 
			(ProcessDefinition)processDefinitionNotationElement.getSemanticElement();
		addNodeGraphicalInfo(processDefinitionNotationElement, nodeElementContainer, processDiagramInfo.elements("node"));
		addContainerGraphicalInfo(processDefinitionNotationElement, nodeElementContainer, processDiagramInfo.elements("node-container"));
		postProcess(processDefinitionNotationElement);
	}

	private void postProcess(NotationElementContainer processDefinitionNotationElement) {
		List nodes = processDefinitionNotationElement.getNodes();
		for (int i = 0; i < nodes.size(); i++) {
			Node node = (Node)nodes.get(i);
			List edges = node.getLeavingEdges();
			for (int j = 0; j < edges.size(); j++) {
				Edge edge = (Edge)edges.get(j);
				NodeElement destination = findDestination(((Transition)edge.getSemanticElement()).getTo(), node);
				Node target = (Node)edge.getFactory().getRegisteredNotationElementFor(destination);
				if (target != null) {
					target.addArrivingEdge(edge);
				}
			}
			if (node instanceof NotationElementContainer) {
				postProcess((NotationElementContainer)node);
			}
		}
	}
	
	private NodeElement findDestination(String path, Node source) {
		NotationElement notationElement = source.getContainer();
		String pathCopy = path;
		while (pathCopy.length() > 3 && "../".equals(pathCopy.substring(0, 3)) && notationElement != null) {
			notationElement = ((Node)notationElement).getContainer();
			pathCopy = pathCopy.substring(3);
		}
		if (notationElement == null) return null;
		JpdlElement parent = (JpdlElement)notationElement.getSemanticElement();
		StringTokenizer tokenizer = new StringTokenizer(pathCopy, "/");
		while (parent != null && tokenizer.hasMoreTokens()) {
			if (!(parent instanceof NodeElementContainer)) return null;
			parent = ((NodeElementContainer)parent).getNodeElementByName(tokenizer.nextToken()); 
		}
		return (NodeElement)parent;
	}
	
	private void addProcessDiagramDimension(
			RootContainer processDefinitionNotationElement, 
			Element processDiagramInfo) {
		String width = processDiagramInfo.attributeValue("width");
		String height = processDiagramInfo.attributeValue("height");
		Dimension dimension = new Dimension(
			width == null ? 0 : Integer.valueOf(width).intValue(),
			height == null ? 0 : Integer.valueOf(height).intValue());
		processDefinitionNotationElement.setDimension(dimension);
	}

	private void addGraphicalInfo(Node node, Element nodeInfo) {
		Rectangle constraint = new Rectangle();
		constraint.x = Integer.valueOf(nodeInfo.attributeValue("x")).intValue();
		constraint.y = Integer.valueOf(nodeInfo.attributeValue("y")).intValue();
		constraint.width = Integer.valueOf(nodeInfo.attributeValue("width")).intValue();
		constraint.height = Integer.valueOf(nodeInfo.attributeValue("height")).intValue();
		node.setConstraint(constraint);
		
		NodeElement semanticElement = (NodeElement)node.getSemanticElement();
		Transition[] transitions = semanticElement.getTransitions();
		List transitionInfoList = nodeInfo.elements("edge");
		int max = Math.min(transitions.length, transitionInfoList.size());
		for (int i = 0; i < max; i++) {
			Edge edge = (Edge)node.getFactory().create("org.jbpm.gd.jpdl.ui.edge");
			edge.setSemanticElement(transitions[i]);
			edge.register();
			node.addLeavingEdge(edge);
			transitions[i].addPropertyChangeListener(edge);
			addGraphicalInfo(
					edge, 
					(Element)transitionInfoList.get(i));
		}
	}

	private void addGraphicalInfo(Edge edge, Element transitionInfo) {
		Element label = transitionInfo.element("label");
		if (label != null) {
			Point offset = new Point();
			offset.x = Integer.valueOf(label.attributeValue("x")).intValue();
			offset.y = Integer.valueOf(label.attributeValue("y")).intValue();
			edge.getLabel().setOffset(offset);
		}
		Iterator bendpointIterator = transitionInfo.elements("bendpoint").iterator();
		while (bendpointIterator.hasNext()) {
			Element bendpointInfo = (Element)bendpointIterator.next();
			edge.addBendPoint(createBendPoint(bendpointInfo)); 
		}
	}
	
	private BendPoint createBendPoint(Element bendpointInfo) {
		BendPoint result = new BendPoint();
		int w1 = Integer.valueOf(bendpointInfo.attributeValue("w1")).intValue();
		int h1 = Integer.valueOf(bendpointInfo.attributeValue("h1")).intValue();
		int w2 = Integer.valueOf(bendpointInfo.attributeValue("w2")).intValue();
		int h2 = Integer.valueOf(bendpointInfo.attributeValue("h2")).intValue();
		Dimension d1 = new Dimension(w1, h1);
		Dimension d2 = new Dimension(w2, h2);
		result.setRelativeDimensions(d1, d2);
		return result;
	}
	
}
