/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.tools;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

import junit.extensions.TestSetup;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.jpdl.par.ProcessArchive;

/**
 * @author Alejandro Guizar
 * @version $Revision: 1.1 $ $Date: 2007/09/05 08:25:11 $
 */
public class WebModuleBuilderTest extends TestCase {

  private static File outputDirectory = new File(System.getProperty("java.io.tmpdir"),
      "WebModuleGenerator");
  private static BpelProcessDefinition processDefinition;
  private static WebModuleBuilder builder;

  public void testCallWsdlServiceGenerator() {
    // service file
    File wsdlDirectory = builder.getWsdlDirectory();
    File serviceFile = new File(wsdlDirectory, builder.getWsdlServiceFileName());
    assertTrue(serviceFile.exists());

    // binding files
    final String bindingFilesPrefix = builder.getWsdlBindingFilesPrefix();
    final String bindingFilesSuffix = ".wsdl";

    File[] bindingFiles = wsdlDirectory.listFiles(new PrefixSuffixFilter(bindingFilesPrefix,
        bindingFilesSuffix));
    assertEquals(1, bindingFiles.length);
  }

  public void testCallJavaMappingGenerator() {
    // jax-rpc mapping file
    File mappingFile = builder.getJaxRpcMappingFile();
    assertTrue(mappingFile.exists());

    // generated classes
    File classesDirectory = builder.getClassesDirectory();
    String[] packageSegments = builder.generateJavaMappingPackage().split("\\.");
    for (int i = 0; i < packageSegments.length; i++) {
      classesDirectory = new File(classesDirectory, packageSegments[i]);
      assertTrue(classesDirectory.exists());
    }
    assertEquals(3, classesDirectory.list().length);
  }

  public void testAddGeneratedFiles() throws IOException {
    ZipFile webModule = new ZipFile(builder.getWebModuleFile());

    try {
      // wsdl files
      File wsdlDirectory = builder.getWsdlDirectory();
      StringBuffer wsdlPathBuffer = new StringBuffer().append(WebModuleBuilder.DIR_WEB_INF).append(
          WebModuleBuilder.SEPARATOR).append(wsdlDirectory.getName()).append(
          WebModuleBuilder.SEPARATOR);
      final int wsdlPathBaseLength = wsdlPathBuffer.length();

      // service file
      String serviceEntry = wsdlPathBuffer.append(builder.getWsdlServiceFileName()).toString();
      assertNotNull(webModule.getEntry(serviceEntry));

      // binding files
      final String bindingFilesPrefix = builder.getWsdlBindingFilesPrefix();
      final String bindingFilesSuffix = ".wsdl";

      File[] bindingFiles = wsdlDirectory.listFiles(new PrefixSuffixFilter(bindingFilesPrefix,
          bindingFilesSuffix));

      for (int i = 0; i < bindingFiles.length; i++) {
        wsdlPathBuffer.setLength(wsdlPathBaseLength);
        String bindingEntry = wsdlPathBuffer.append(bindingFiles[i].getName()).toString();
        assertNotNull(webModule.getEntry(bindingEntry));
      }

      // jax-rpc mapping file
      String jaxRpcMappingEntry = WebModuleBuilder.DIR_WEB_INF
          + WebModuleBuilder.SEPARATOR
          + builder.getJaxRpcMappingFile().getName();
      assertNotNull(webModule.getEntry(jaxRpcMappingEntry));

      // classes
      String classesPath = WebModuleBuilder.DIR_WEB_INF
          + WebModuleBuilder.SEPARATOR
          + WebModuleBuilder.DIR_CLASSES
          + WebModuleBuilder.SEPARATOR
          + builder.generateJavaMappingPackage().replace('.', WebModuleBuilder.SEPARATOR);
      int classCount = 0;
      for (Enumeration entries = webModule.entries(); entries.hasMoreElements();) {
        ZipEntry entry = (ZipEntry) entries.nextElement();
        if (entry.getName().startsWith(classesPath))
          classCount++;
      }
      assertEquals(3, classCount);
    }
    finally {
      webModule.close();
    }
  }

  public void testAddUserProvidedFiles() throws IOException {
    ZipFile webModule = new ZipFile(builder.getWebModuleFile());

    // web.xml
    String webEntry = WebModuleBuilder.DIR_WEB_INF + WebModuleBuilder.SEPARATOR + "web.xml";
    assertNotNull(webModule.getEntry(webEntry));

    // webservices.xml
    String webservicesEntry = WebModuleBuilder.DIR_WEB_INF
        + WebModuleBuilder.SEPARATOR
        + "webservices.xml";
    assertNotNull(webModule.getEntry(webservicesEntry));

    // bpel-application.xml
    String bpelApplicationEntry = WebModuleBuilder.DIR_WEB_INF
        + WebModuleBuilder.SEPARATOR
        + "classes"
        + WebModuleBuilder.SEPARATOR
        + "bpel-application.xml";
    assertNotNull(webModule.getEntry(bpelApplicationEntry));
  }

  public static Test suite() {
    return new Setup();
  }

  private static class PrefixSuffixFilter implements FilenameFilter {

    private final String ext;
    private final String base;

    PrefixSuffixFilter(String base, String ext) {
      this.ext = ext;
      this.base = base;
    }

    public boolean accept(File dir, String name) {
      return name.startsWith(base) && name.endsWith(ext);
    }
  }

  private static class Setup extends TestSetup {

    Setup() {
      super(new TestSuite(WebModuleBuilderTest.class));
    }

    protected void setUp() throws Exception {
      // build process archive
      URL processLocation = WebModuleBuilderTest.class.getResource("hello-process.zip");
      assertNotNull(processLocation);

      ProcessArchive processArchive;
      InputStream processStream = processLocation.openStream();
      try {
        processArchive = new ProcessArchive(new ZipInputStream(processStream));
      }
      finally {
        processStream.close();
      }

      // read process definition
      JbpmContext jbpmContext = JbpmConfiguration.getInstance().createJbpmContext();
      try {
        processDefinition = (BpelProcessDefinition) processArchive.parseProcessDefinition();
      }
      finally {
        jbpmContext.close();
      }

      // prepare output directory
      if (outputDirectory.exists())
        FileUtil.deleteDirectory(outputDirectory);

      outputDirectory.mkdir();

      // call web module generator
      builder = new WebModuleBuilder();
      builder.setOutputDirectory(outputDirectory);
      builder.generateWebModule(processDefinition);
      assertEquals(0, builder.getProblemHandler().getProblemCount());
    }

    protected void tearDown() throws Exception {
      // comment if you want to see the generated files
      FileUtil.deleteDirectory(outputDirectory);
    }
  }
}
