/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.app;

import java.io.IOException;
import java.io.StringReader;
import java.net.URISyntaxException;
import java.util.Map;

import javax.xml.transform.dom.DOMSource;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.bpel.BpelException;
import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.bpel.graph.scope.Scope;
import org.jbpm.bpel.graph.struct.Sequence;
import org.jbpm.bpel.xml.AppDescriptorReader;
import org.jbpm.bpel.xml.BpelConstants;
import org.jbpm.bpel.xml.BpelReader;
import org.jbpm.bpel.xml.util.XmlUtil;

/**
 * @author Juan Cant
 * @version $Revision: 1.15 $ $Date: 2007/08/21 09:33:20 $
 */
public class ScopeMatcherTest extends TestCase {

  private AppDescriptorReader appDescriptorReader = new AppDescriptorReader();
  private BpelProcessDefinition processDefinition;
  private ScopeMatcher scopeMatcher;

  private JbpmContext jbpmContext;

  private static DOMSource processSource;

  protected void setUp() throws Exception {
    /*
     * the reader accesses the jbpm configuration, so create a context before creating the reader to
     * avoid loading another configuration from the default resource
     */
    JbpmConfiguration jbpmConfiguration = JbpmConfiguration.getInstance("org/jbpm/bpel/graph/exe/test.jbpm.cfg.xml");
    jbpmContext = jbpmConfiguration.createJbpmContext();

    processDefinition = new BpelProcessDefinition();
    new BpelReader().read(processDefinition, processSource);
    scopeMatcher = new ScopeMatcher(processDefinition);
  }

  protected void tearDown() throws Exception {
    jbpmContext.close();
  }

  public void testMatchProcessDescriptor() throws Exception {
    String text = "<bpelApplication xmlns='"
        + BpelConstants.NS_APP_DESCRIPTOR
        + "'>"
        + " <partnerLinks>"
        + "  <partnerLink name='schedulingPL'>"
        + "   <myRole handle='globalScheduling'/>"
        + "  </partnerLink>"
        + " </partnerLinks>"
        + "</bpelApplication>";

    AppDescriptor appDescriptor = new AppDescriptor();
    appDescriptorReader.read(appDescriptor, new InputSource(new StringReader(text)));

    scopeMatcher.visit(appDescriptor);
    Map scopeDescriptors = scopeMatcher.getScopeDescriptors();

    assertSame(appDescriptor, scopeDescriptors.get(processDefinition.getGlobalScope()));
  }

  public void testMatchScopeDescriptor() throws Exception {
    String text = "<bpelApplication xmlns='"
        + BpelConstants.NS_APP_DESCRIPTOR
        + "'>"
        + " <scopes>"
        + "  <scope name='s1'>"
        + "   <scopes>"
        + "    <scope name='s1.1'/>"
        + "   </scopes>"
        + "  </scope>"
        + " </scopes>"
        + "</bpelApplication>";

    AppDescriptor appDescriptor = new AppDescriptor();
    appDescriptorReader.read(appDescriptor, new InputSource(new StringReader(text)));

    Sequence sequence = (Sequence) processDefinition.getGlobalScope().getActivity();
    Scope s1Def = (Scope) sequence.getNodes().get(1);
    AppScope s1Config = (AppScope) appDescriptor.getScopes().get(0);

    Scope s11Def = (Scope) s1Def.getActivity();
    AppScope s11Config = (AppScope) s1Config.getScopes().get(0);

    scopeMatcher.visit(appDescriptor);
    Map scopeDescriptors = scopeMatcher.getScopeDescriptors();

    assertSame(s1Config, scopeDescriptors.get(s1Def));
    assertSame(s11Config, scopeDescriptors.get(s11Def));
  }

  public void testMatchUnnamedScopeDescriptor() throws Exception {
    String text = "<bpelApplication xmlns='"
        + BpelConstants.NS_APP_DESCRIPTOR
        + "'>"
        + " <scopes>"
        + "  <scope />"
        + " </scopes>"
        + "</bpelApplication>";

    AppDescriptor appDescriptor = new AppDescriptor();
    appDescriptorReader.read(appDescriptor, new InputSource(new StringReader(text)));
    AppScope s1Descriptor = (AppScope) appDescriptor.getScopes().get(0);

    scopeMatcher.visit(appDescriptor);
    Map scopeDescriptors = scopeMatcher.getScopeDescriptors();

    Sequence sequence = (Sequence) processDefinition.getGlobalScope().getActivity();
    Scope s1Def = (Scope) sequence.getNodes().get(2);

    assertSame(s1Descriptor, scopeDescriptors.get(s1Def));
  }

  public void testConflictingScopeDescriptor() throws Exception {
    String text = "<bpelApplication xmlns='"
        + BpelConstants.NS_APP_DESCRIPTOR
        + "'>"
        + " <scopes>"
        + "  <scope name='conflictingName' />"
        + " </scopes>"
        + "</bpelApplication>";

    AppDescriptor appDescriptor = new AppDescriptor();
    appDescriptorReader.read(appDescriptor, new InputSource(new StringReader(text)));

    try {
      scopeMatcher.visit(appDescriptor);
      fail("conflicting name resolution must fail");
    }
    catch (BpelException e) {
      // this exception is expected
    }
  }

  public static Test suite() throws SAXException, IOException, URISyntaxException {
    processSource = XmlUtil.parseResource("mergeProcess.bpel", ScopeMatcherTest.class);
    return new TestSuite(ScopeMatcherTest.class);
  }
}
