/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.integration.jms;

import java.util.Map;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.QueueReceiver;
import javax.jms.Session;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.StaleStateException;
import org.hibernate.exception.LockAcquisitionException;

import org.jbpm.JbpmContext;
import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.bpel.graph.exe.BpelFaultException;
import org.jbpm.bpel.integration.def.ReceiveAction;
import org.jbpm.bpel.persistence.db.IntegrationSession;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.graph.exe.Token;

/**
 * @author Alejandro Guzar
 * @version $Revision: 1.16 $ $Date: 2007/08/08 10:50:48 $
 */
public class StartListener implements MessageListener {

  private final long processDefinitionId;
  private final long receiveActionId;

  private final IntegrationControl integrationControl;
  private final Session jmsSession;
  private final MessageConsumer consumer;

  private static final Log log = LogFactory.getLog(StartListener.class);

  StartListener(BpelProcessDefinition processDefinition, ReceiveAction receiveAction,
      IntegrationControl integrationControl, Session jmsSession) throws JMSException {
    this.processDefinitionId = processDefinition.getId();
    this.receiveActionId = receiveAction.getId();

    this.integrationControl = integrationControl;
    this.jmsSession = jmsSession;

    // get the destination associated to the partner link
    Destination destination = integrationControl.getPartnerLinkEntry(receiveAction.getPartnerLink())
        .getDestination();
    // format the message selector
    String selector = formatSelector(receiveAction);
    // create message consumer for destination and selector above
    consumer = jmsSession.createConsumer(destination, selector);

    log.debug("created start listener: processDefinition="
        + processDefinition
        + ", receiveAction="
        + receiveAction);
  }

  private static String formatSelector(ReceiveAction receiveAction) {
    return IntegrationConstants.PARTNER_LINK_ID_PROP
        + '='
        + receiveAction.getPartnerLink().getId()
        + " AND "
        + IntegrationConstants.OPERATION_NAME_PROP
        + "='"
        + receiveAction.getOperation().getName()
        + '\'';
  }

  public void open() throws JMSException {
    /*
     * jms could deliver a message immediately after setting this listener, so make sure this
     * listener is fully initialized at this point
     */
    consumer.setMessageListener(this);
    log.debug("opened start listener: process="
        + processDefinitionId
        + ", receiver="
        + receiveActionId);
  }

  public long getReceiveActionId() {
    return receiveActionId;
  }

  public void onMessage(Message message) {
    if (!(message instanceof ObjectMessage)) {
      log.error("received non-object jms message: " + message);
      return;
    }
    ObjectMessage request = (ObjectMessage) message;
    try {
      log.debug("delivering request: " + RequestListener.messageToString(request));
      deliverRequest((Map) request.getObject(), request.getJMSReplyTo(), request.getJMSMessageID());
      request.acknowledge();
    }
    catch (JMSException e) {
      log.error("request delivery failed due to jms exception, giving up", e);
    }
    catch (RuntimeException e) {
      if (isRecoverable(e)) {
        log.warn("request delivery failed due to recoverable exception, attempting recovery");
        try {
          // recover the session manually
          jmsSession.recover();
        }
        catch (JMSException je) {
          log.error("request recovery failed, giving up", je);
        }
      }
      else
        log.error("request delivery failed due to non-recoverable exception, giving up", e);
    }
  }

  private static boolean isRecoverable(RuntimeException exception) {
    for (Throwable throwable = exception; throwable != null; throwable = throwable.getCause()) {
      if (throwable instanceof StaleStateException || throwable instanceof LockAcquisitionException)
        return true;
    }
    return false;
  }

  private void deliverRequest(Map parts, Destination replyTo, String requestId) {
    JbpmContext jbpmContext = integrationControl.getIntegrationServiceFactory()
        .getJbpmConfiguration()
        .createJbpmContext();
    try {
      // load process definition
      BpelProcessDefinition processDefinition = (BpelProcessDefinition) jbpmContext.getSession()
          .load(BpelProcessDefinition.class, new Long(processDefinitionId));
      // load the message receiver
      ReceiveAction receiveAction = IntegrationSession.getInstance(jbpmContext).loadReceiveAction(
          receiveActionId);

      // instantiate the process
      ProcessInstance processInstance = new ProcessInstance(processDefinition);
      // XXX root token is not assigned an identifier at creation time
      jbpmContext.getServices().getPersistenceService().assignId(processInstance.getRootToken());

      // build initial runtime structures
      Token receivingToken = receiveAction.initializeProcessInstance(processInstance);

      try {
        // file outstanding request, in case operation has output
        if (receiveAction.getOperation().getOutput() != null) {
          // encapsulate the fields needed to reply
          OutstandingRequest outRequest = new OutstandingRequest(replyTo, requestId);
          // register the request in the integration control
          integrationControl.addOutstandingRequest(receiveAction, receivingToken, outRequest);
        }

        // pass control to start activity
        receiveAction.deliverMessage(receivingToken, parts);
      }
      catch (BpelFaultException e) {
        log.debug("request delivery caused a fault", e);
        processDefinition.getGlobalScope().raiseException(e, new ExecutionContext(receivingToken));
      }
      // save changes to instance
      jbpmContext.save(processInstance);
    }
    catch (RuntimeException e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    finally {
      // end transaction, close all services
      jbpmContext.close();
    }
  }

  public void close() throws JMSException {
    consumer.close();
    log.debug("closed start listener: process="
        + processDefinitionId
        + ", receiver="
        + receiveActionId);
  }

  public String toString() {
    ToStringBuilder builder = new ToStringBuilder(this);
    try {
      builder.append("queue", ((QueueReceiver) consumer).getQueue()).append("selector",
          consumer.getMessageSelector());
    }
    catch (JMSException e) {
      log.debug("could not fill request listener fields", e);
    }
    return builder.toString();
  }
}
