/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.integration.jms;

import java.util.Collections;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.JMSException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;
import javax.wsdl.Definition;
import javax.wsdl.Operation;
import javax.wsdl.Port;
import javax.xml.namespace.QName;

import org.apache.commons.collections.LRUMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmContext;
import org.jbpm.bpel.BpelException;
import org.jbpm.bpel.app.AppDescriptor;
import org.jbpm.bpel.app.ScopeMatcher;
import org.jbpm.bpel.app.AppPartnerRole.InitiateMode;
import org.jbpm.bpel.endpointref.EndpointReference;
import org.jbpm.bpel.endpointref.EndpointReferenceFactory;
import org.jbpm.bpel.graph.basic.Receive;
import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.bpel.graph.exe.BpelFaultException;
import org.jbpm.bpel.graph.scope.OnEvent;
import org.jbpm.bpel.graph.scope.Scope;
import org.jbpm.bpel.graph.struct.Pick;
import org.jbpm.bpel.graph.struct.StructuredActivity.Begin;
import org.jbpm.bpel.integration.catalog.ServiceCatalog;
import org.jbpm.bpel.integration.client.SoapClient;
import org.jbpm.bpel.integration.def.PartnerLinkDefinition;
import org.jbpm.bpel.integration.def.ReceiveAction;
import org.jbpm.bpel.integration.def.ReplyAction;
import org.jbpm.bpel.integration.exe.PartnerLinkInstance;
import org.jbpm.bpel.persistence.db.IntegrationSession;
import org.jbpm.bpel.xml.BpelConstants;
import org.jbpm.graph.exe.Token;
import org.jbpm.svc.Services;

/**
 * @author Alejandro Guzar
 * @version $Revision: 1.17 $ $Date: 2007/07/26 00:39:12 $
 */
public class IntegrationControl {

  private final JmsIntegrationServiceFactory integrationServiceFactory;

  private Connection jmsConnection;
  private AppDescriptor appDescriptor;

  private List partnerLinkEntries = Collections.EMPTY_LIST;

  private List startListeners = Collections.EMPTY_LIST;
  private final Map requestListeners = new Hashtable();
  private final Map outstandingRequests = new Hashtable();

  private Definition wsdlDefinition;

  private final Map partnerClients = new LRUMap();

  /**
   * Default JNDI subcontext containing JMS objects, relative to the initial context.
   */
  public static final String DEFAULT_JMS_CONTEXT = "java:comp/env/jms";

  /** JNDI name bound to the JMS connection factory, relative to the JMS context. */
  public static final String CONNECTION_FACTORY_NAME = "IntegrationConnectionFactory";

  private static final Log log = LogFactory.getLog(IntegrationControl.class);
  private static final long serialVersionUID = 1L;

  IntegrationControl(JmsIntegrationServiceFactory integrationServiceFactory) {
    this.integrationServiceFactory = integrationServiceFactory;
  }

  public JmsIntegrationServiceFactory getIntegrationServiceFactory() {
    return integrationServiceFactory;
  }

  public Connection getJmsConnection() {
    return jmsConnection;
  }

  public AppDescriptor getAppDescriptor() {
    return appDescriptor;
  }

  public void setAppDescriptor(AppDescriptor appDescriptor) {
    this.appDescriptor = appDescriptor;
  }

  public PartnerLinkEntry getPartnerLinkEntry(PartnerLinkDefinition partnerLink) {
    final long partnerLinkId = partnerLink.getId();
    for (int i = 0, n = partnerLinkEntries.size(); i < n; i++) {
      PartnerLinkEntry entry = (PartnerLinkEntry) partnerLinkEntries.get(i);
      if (entry.getId() == partnerLinkId)
        return entry;
    }
    return null;
  }

  public PartnerLinkEntry getPartnerLinkEntry(QName portTypeName, QName serviceName, String portName) {
    PartnerLinkEntry selectedEntry = null;

    for (int i = 0, n = partnerLinkEntries.size(); i < n; i++) {
      PartnerLinkEntry entry = (PartnerLinkEntry) partnerLinkEntries.get(i);
      EndpointReference myReference = entry.getMyReference();

      if (myReference == null)
        continue; // this entry corresponds to a partner link without myRole

      QName myPortTypeName = myReference.getPortTypeName();
      if (!portTypeName.equals(myPortTypeName))
        continue;

      QName myServiceName = myReference.getServiceName();
      if (myServiceName != null && !serviceName.equals(myServiceName))
        continue;

      String myPortName = myReference.getPortName();
      if (myPortName != null && !portName.equals(myPortName))
        continue;

      if (selectedEntry != null) {
        throw new BpelException("multiple partner link entries match the given arguments: service="
            + serviceName
            + ", port="
            + portName);
      }
      selectedEntry = entry;
    }
    return selectedEntry;
  }

  public Map getRequestListeners() {
    return requestListeners;
  }

  public void addRequestListener(RequestListener requestListener) {
    Object key = createKey(requestListener.getReceiveActionId(), requestListener.getTokenId());
    requestListeners.put(key, requestListener);
  }

  public RequestListener removeRequestListener(ReceiveAction receiveAction, Token token) {
    Object key = createKey(receiveAction.getId(), token.getId());
    return (RequestListener) requestListeners.remove(key);
  }

  public void removeRequestListener(RequestListener requestListener) {
    Object key = createKey(requestListener.getReceiveActionId(), requestListener.getTokenId());
    synchronized (requestListeners) {
      Object currentListener = requestListeners.get(key);
      if (requestListener == currentListener)
        requestListeners.remove(key);
    }
  }

  private static Object createKey(long receiveActionId, long tokenId) {
    return new RequestListener.Key(receiveActionId, tokenId);
  }

  public Map getOutstandingRequests() {
    return outstandingRequests;
  }

  public void addOutstandingRequest(ReceiveAction receiveAction, Token token,
      OutstandingRequest request) {
    Object key = createKey(receiveAction.getPartnerLink().getInstance(token),
        receiveAction.getOperation(), receiveAction.getMessageExchange());

    if (outstandingRequests.put(key, request) != null)
      throw new BpelFaultException(BpelConstants.FAULT_CONFLICTING_REQUEST);

    log.debug("added outstanding request: receiver="
        + receiveAction
        + ", token="
        + token
        + ", request="
        + request);
  }

  public OutstandingRequest removeOutstandingRequest(ReplyAction replyAction, Token token) {
    Object key = createKey(replyAction.getPartnerLink().getInstance(token),
        replyAction.getOperation(), replyAction.getMessageExchange());
    OutstandingRequest request = (OutstandingRequest) outstandingRequests.remove(key);
    if (request == null)
      throw new BpelFaultException(BpelConstants.FAULT_MISSING_REQUEST);

    log.debug("removed outstanding request: replier="
        + replyAction
        + ", token="
        + token
        + ", request="
        + request);
    return request;
  }

  private static Object createKey(PartnerLinkInstance partnerLinkInstance, Operation operation,
      String messageExchange) {
    return new OutstandingRequest.Key(getOrAssignId(partnerLinkInstance), operation.getName(),
        messageExchange);
  }

  public Map getPartnerClients() {
    return partnerClients;
  }

  public SoapClient getPartnerClient(PartnerLinkInstance instance) {
    Long instanceId = new Long(getOrAssignId(instance));
    synchronized (partnerClients) {
      // retrieve cached port consumer
      SoapClient partnerClient = (SoapClient) partnerClients.get(instanceId);
      // cache miss?
      if (partnerClient == null) {
        // create client from partner endpoint reference
        partnerClient = createPartnerClient(instance);
        partnerClients.put(instanceId, partnerClient);
      }
      return partnerClient;
    }
  }

  private SoapClient createPartnerClient(PartnerLinkInstance instance) {
    EndpointReference partnerRef = instance.getPartnerReference();
    if (partnerRef == null) {
      // no partner reference, create one containing only the port type as
      // selection criterion
      partnerRef = createPartnerReference(instance.getDefinition());
      instance.setPartnerReference(partnerRef);
      log.debug("initialized partner reference: instance=" + instance + ", reference=" + partnerRef);
    }
    // select a port from the service catalog with the criteria known at
    // this point
    Port port = partnerRef.selectPort(getServiceCatalog());
    log.debug("selected partner port: instance=" + instance + ", port=" + port.getName());
    // create a client for that port
    return new SoapClient(port);
  }

  private static long getOrAssignId(PartnerLinkInstance instance) {
    long instanceId = instance.getId();
    // in case instance is transient, assign an identifier to it
    if (instanceId == 0L) {
      Services.assignId(instance);
      instanceId = instance.getId();
    }
    return instanceId;
  }

  EndpointReference createPartnerReference(PartnerLinkDefinition definition) {
    PartnerLinkEntry entry = getPartnerLinkEntry(definition);
    InitiateMode initiateMode = entry.getInitiateMode();

    EndpointReference partnerReference;
    if (InitiateMode.STATIC.equals(initiateMode)) {
      partnerReference = entry.getPartnerReference();
    }
    else if (InitiateMode.PULL.equals(initiateMode)) {
      EndpointReferenceFactory referenceFactory = EndpointReferenceFactory.getInstance(
          IntegrationConstants.DEFAULT_REFERENCE_NAME, null);
      partnerReference = referenceFactory.createEndpointReference();
    }
    else
      throw new BpelFaultException(BpelConstants.FAULT_UNINITIALIZED_PARTNER_ROLE);

    return partnerReference;
  }

  public ServiceCatalog getServiceCatalog() {
    return getAppDescriptor().getServiceCatalog();
  }

  public List getStartListeners() {
    return startListeners;
  }

  public Definition getWsdlDefinition() {
    return wsdlDefinition;
  }

  public void setWsdlDefinition(Definition wsdlDefinition) {
    this.wsdlDefinition = wsdlDefinition;
  }

  /**
   * Prepares inbound message activities annotated to create a process instance for receiving
   * requests.
   */
  public void enableInboundMessageActivities(JbpmContext jbpmContext) throws NamingException,
      JMSException {
    InitialContext initialContext = new InitialContext();
    try {
      // publish partner link information to JNDI
      BpelProcessDefinition processDefinition = getAppDescriptor().findProcessDefinition(
          jbpmContext);
      buildPartnerLinkEntries(initialContext, processDefinition);

      // open a jms connection
      openJmsConnection(initialContext);

      try {
        // enable start IMAs
        StartListenersBuilder builder = new StartListenersBuilder(this);
        builder.visit(processDefinition);
        startListeners = builder.getStartListeners();

        if (startListeners.isEmpty())
          throw new BpelException(processDefinition + " has no start activities");

        for (int i = 0, n = startListeners.size(); i < n; i++) {
          StartListener startListener = (StartListener) startListeners.get(i);
          startListener.open();
        }

        // enable outstanding IMAs
        IntegrationSession integrationSession = IntegrationSession.getInstance(jbpmContext);
        JmsIntegrationService integrationService = JmsIntegrationService.get(jbpmContext);

        // receive
        Iterator receiveTokenIt = integrationSession.findReceiveTokens(processDefinition)
            .iterator();
        while (receiveTokenIt.hasNext()) {
          Token token = (Token) receiveTokenIt.next();
          Receive receive = (Receive) token.getNode();
          integrationService.jmsReceive(receive.getReceiveAction(), token, this, true);
        }

        // pick
        Iterator pickTokenIt = integrationSession.findPickTokens(processDefinition).iterator();
        while (pickTokenIt.hasNext()) {
          Token token = (Token) pickTokenIt.next();
          // pick points activity token to begin mark
          Begin begin = (Begin) token.getNode();
          Pick pick = (Pick) begin.getCompositeActivity();
          integrationService.jmsReceive(pick.getOnMessages(), token, this);
        }

        // event
        Iterator eventTokenIt = integrationSession.findEventTokens(processDefinition).iterator();
        while (eventTokenIt.hasNext()) {
          Token token = (Token) eventTokenIt.next();
          // scope points events token to itself
          Scope scope = (Scope) token.getNode();
          Iterator onEventsIt = scope.getOnEvents().iterator();
          while (onEventsIt.hasNext()) {
            OnEvent onEvent = (OnEvent) onEventsIt.next();
            integrationService.jmsReceive(onEvent.getReceiveAction(), token, this, false);
          }
        }

        // start message delivery
        jmsConnection.start();
      }
      catch (JMSException e) {
        jmsConnection.close();
        throw e;
      }
    }
    finally {
      initialContext.close();
    }
  }

  /**
   * Prevents inbound message activities annotated to create a process instance from further
   * receiving requests.
   */
  public void disableInboundMessageActivities() throws JMSException {
    // disable start IMAs
    Iterator startListenerIt = startListeners.iterator();
    while (startListenerIt.hasNext()) {
      StartListener startListener = (StartListener) startListenerIt.next();
      startListener.close();
    }
    startListeners = Collections.EMPTY_LIST;

    // disable outstanding IMAs
    synchronized (requestListeners) {
      Iterator requestListenerIt = requestListeners.values().iterator();
      while (requestListenerIt.hasNext()) {
        RequestListener requestListener = (RequestListener) requestListenerIt.next();
        requestListener.close();
      }
      requestListeners.clear();
    }

    // release jms connection
    closeJmsConnection();
  }

  void buildPartnerLinkEntries(InitialContext initialContext, BpelProcessDefinition process)
      throws NamingException {
    // match scopes with their descriptors
    ScopeMatcher scopeMatcher = new ScopeMatcher(process);
    scopeMatcher.visit(getAppDescriptor());
    Map scopeDescriptors = scopeMatcher.getScopeDescriptors();
    // lookup destinations & bind port entries
    PartnerLinkEntriesBuilder builder = new PartnerLinkEntriesBuilder(scopeDescriptors,
        getJmsContext(initialContext), integrationServiceFactory.getDefaultDestination());
    builder.visit(process);
    partnerLinkEntries = builder.getPartnerLinkEntries();
  }

  void openJmsConnection(InitialContext initialContext) throws NamingException, JMSException {
    ConnectionFactory jmsConnectionFactory = getConnectionFactory(initialContext);
    jmsConnection = jmsConnectionFactory.createConnection();
  }

  private ConnectionFactory getConnectionFactory(InitialContext initialContext)
      throws NamingException, NameNotFoundException {
    Context jmsContext = getJmsContext(initialContext);
    ConnectionFactory jmsConnectionFactory;
    try {
      jmsConnectionFactory = (ConnectionFactory) jmsContext.lookup(CONNECTION_FACTORY_NAME);
      log.debug("retrieved jms connection factory: " + CONNECTION_FACTORY_NAME);
    }
    catch (NameNotFoundException e) {
      log.debug("jms connection factory not found: " + CONNECTION_FACTORY_NAME);
      log.debug("falling back to default from integration service factory");
      jmsConnectionFactory = integrationServiceFactory.getDefaultConnectionFactory();
      if (jmsConnectionFactory == null)
        throw e;
    }
    return jmsConnectionFactory;
  }

  void closeJmsConnection() throws JMSException {
    if (jmsConnection != null) {
      jmsConnection.close();
      jmsConnection = null;
    }
  }

  void reset() {
    appDescriptor = null;

    partnerLinkEntries = Collections.EMPTY_LIST;

    startListeners = Collections.EMPTY_LIST;
    requestListeners.clear();
    outstandingRequests.clear();

    partnerClients.clear();
  }

  static Context getJmsContext(InitialContext initialContext) {
    Context jmsContext;
    try {
      jmsContext = (Context) initialContext.lookup(DEFAULT_JMS_CONTEXT);
    }
    catch (NamingException e) {
      log.debug("could not retrieve jms context, falling back to initial context");
      jmsContext = initialContext;
    }
    return jmsContext;
  }
}
