/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2007,
 * @author JBoss Inc.
 */
package org.jboss.soa.esb.quickstart.test;

import java.lang.reflect.UndeclaredThrowableException;

import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.jboss.soa.esb.server.QuickstartMessageStore;
import org.jboss.soa.esb.server.QuickstartMessageStoreImplMBean;
import org.jboss.test.JBossTestCase;

/**
 * Abstract test case for Quickstart tests
 * 
 * @author <a href="mailto:kconner@redhat.com">Kevin Conner</a>
 */
public abstract class AbstractQuickstartTestCase extends JBossTestCase
{
    /**
     * Construct the test case with the specified name.
     * @param name The test name.
     */
    protected AbstractQuickstartTestCase(final String name)
    {
        super(name) ;
    }
    
    /**
     * Clear the messages in the sink.
     * @throws Exception for unexpected errors.
     */
    protected void clearMessages()
        throws Exception
    {
        getQuickstartMessageStore().clearMessages() ;
    }
    
    /**
     * Wait for the specified count of messages.
     * @param numberOfMessages The expected number of messages.
     * @param maxDelay The maximum delay to wait for the messages.
     * @return The message list if present, otherwise current contents.
     * @throws Exception for unexpected errors.
     */
    protected String[] waitForMessages(final int numberOfMessages, final long maxDelay)
        throws Exception
    {
        final String[] sinkMessages = getQuickstartMessageStore().waitForMessages(numberOfMessages, maxDelay) ;
        final int numberOfSinkMessages = (sinkMessages == null ? 0 : sinkMessages.length) ;
        assertEquals("Message count", numberOfMessages, numberOfSinkMessages) ;
        return sinkMessages ;
    }
    
    /**
     * Check the stored messages against the expected messages.
     * @param maxDelay The maximum delay to wait for the messages.
     * @param expectedMessages The expected messages.
     * @throws Exception for unexpected errors.
     */
    protected void checkMessages(final long maxDelay, String ... expectedMessages)
        throws Exception
    {
        final int numberOfMessages = (expectedMessages == null ? 0 : expectedMessages.length) ;
        final String[] sinkMessages = waitForMessages(numberOfMessages, maxDelay) ;
        
        for(int count = 0 ; count < numberOfMessages ; count++)
        {
            assertEquals("Message: " + count, expectedMessages[count],
                sinkMessages[count]) ;
        }
    }
    
    /**
     * Check the stored messages against the expected patterns.
     * @param maxDelay The maximum delay to wait for the messages.
     * @param expectedPatterns The patterns of the expected messages.
     * @throws Exception for unexpected errors.
     */
    protected void checkPatternMessages(final long maxDelay, String ... expectedPatterns)
        throws Exception
    {
        final int numberOfMessages = (expectedPatterns == null ? 0 : expectedPatterns.length) ;
        final String[] sinkMessages = waitForMessages(numberOfMessages, maxDelay) ;
        for(int count = 0 ; count < numberOfMessages ; count++)
        {
            final String sinkMessage = sinkMessages[count] ;
            assertTrue("Message: " + count, (sinkMessage != null) && sinkMessage.matches(expectedPatterns[count])) ;
        }
    }
    
    /**
     * Get a proxy to the quickstart message store MBean.
     * @return The mbean proxy.
     * @throws Exception for unexpected errors.
     */
    private QuickstartMessageStore getQuickstartMessageStore()
        throws Exception
    {
        return new QuickstartMessageStoreProxy(getServer()) ;
    }
    
    /**
     * Proxy class to simplify the interactions with the MBean.
     * @author kevin
     */
    private static final class QuickstartMessageStoreProxy implements QuickstartMessageStore
    {
        private final MBeanServerConnection serverConnection ;
        
        QuickstartMessageStoreProxy(final MBeanServerConnection serverConnection)
        {
            this.serverConnection = serverConnection ;
        }

        public void clearMessages()
        {
            try
            {
                serverConnection.invoke(new ObjectName(QuickstartMessageStoreImplMBean.objectName), "clearMessages",
                    null, null) ;
            }
            catch (final RuntimeException re)
            {
                throw re ;
            }
            catch (final Error error)
            {
                throw error ;
            }
            catch (final Throwable th)
            {
                throw new UndeclaredThrowableException(th) ;
            }
        }
        
        public String[] waitForMessages(int numberOfMessages, long maxDelay)
        {
            try
            {
                return (String[])serverConnection.invoke(new ObjectName(QuickstartMessageStoreImplMBean.objectName), "waitForMessages",
                    new Object[] {Integer.valueOf(numberOfMessages), Long.valueOf(maxDelay)},
                    new String[] {Integer.TYPE.getName(), Long.TYPE.getName()}) ;
            }
            catch (final RuntimeException re)
            {
                throw re ;
            }
            catch (final Error error)
            {
                throw error ;
            }
            catch (final Throwable th)
            {
                throw new UndeclaredThrowableException(th) ;
            }
        }
    }
}
