/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.server.jbpmAsync;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.jbpm.graph.def.ActionHandler;
import org.jbpm.graph.exe.ExecutionContext;

/**
 * Action Handler for jBPM Async test.
 */
public class JbpmAsyncActionHandler implements ActionHandler
{
    /**
     * Serial version UID for this class.
     */
    private static final long serialVersionUID = 6205275164157280372L;

    /**
     * The notification lock.
     */
    private static final Lock NOTIFICATION_LOCK = new ReentrantLock() ;
    
    /**
     * The notification condition.
     */
    private static Condition NOTIFICATION_CONDITION = NOTIFICATION_LOCK.newCondition() ;
    
    /**
     * The waiting flag.
     */
    private static boolean waiting ;
    /**
     * The notified flag.
     */
    private static boolean notified ;
    
    /**
     * Wait for a notification from the test.
     */
    public void execute(final ExecutionContext executionContext)
        throws Exception
    {
        if (!waitForNotification(60000L))
        {
            throw new Exception("Notification failed") ;
        }
    }
    
    /**
     * Notify the action.
     * @param delay The maximum delay waiting for the action.
     * @return true if waiting, false otherwise.
     */
    public static boolean notifyAction(final long delay)
    {
        NOTIFICATION_LOCK.lock() ;
        try
        {
            if (!waiting)
            {
                try
                {
                    NOTIFICATION_CONDITION.await(delay, TimeUnit.MILLISECONDS) ;
                }
                catch (final InterruptedException ie) {}
            }
            
            if (waiting)
            {
                notified = true ;
                NOTIFICATION_CONDITION.signal() ;
            }
            return waiting ;
        }
        finally
        {
            NOTIFICATION_LOCK.unlock() ;
        }
    }
    
    /**
     * Wait for a notification from the test.
     * @param delay The maximum delay waiting for notification.
     * @return true if notified, false otherwise.
     */
    private static boolean waitForNotification(final long delay)
    {
        NOTIFICATION_LOCK.lock() ;
        waiting = true ;
        try
        {
            NOTIFICATION_CONDITION.signal() ;
            try
            {
                NOTIFICATION_CONDITION.await(delay, TimeUnit.MILLISECONDS) ;
            }
            catch (final InterruptedException ie) {}
            return notified ;
        }
        finally
        {
            waiting = false ;
            notified = false ;
            NOTIFICATION_LOCK.unlock() ;
        }
    }
}
