/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2007
 */

package org.jboss.soa.esb.eclipse.wizards;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Properties;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.jboss.soa.esb.eclipse.template.Activator;
import org.jboss.soa.esb.eclipse.util.ESBUtils;

/**
 * This object is a helper class which helps create the java project by
 * copying in the template files, and create the classpath using the 
 */
public class ESBProjectCreator {
	private static final String TEMPLATE_FILES_DIR = "/template";
	
        private static final String QUICKSTART_CONF_PROPERTY = "quickstart.conf" ;
        
	/**
	 * Copy all of the template files from the plugin into the new eclipse 
	 * project.
	 */
	public void copyTemplateFiles(IProject f_project, final IProgressMonitor monitor) throws CoreException, IOException {
            final IPath templatePath = Path.fromOSString(TEMPLATE_FILES_DIR) ;
            final Activator activator = Activator.getDefault() ;
            final URL templateURL = FileLocator.find(activator.getBundle(), templatePath, null) ;
            final URL templateFileURL = FileLocator.toFileURL(templateURL) ;
            final URI templateURI ;
            try
            {
                templateURI = new URI(templateFileURL.toExternalForm()) ;
            }
            catch (final URISyntaxException urise)
            {
                throw new CoreException(new Status(Status.ERROR, Activator.PLUGIN_ID, 0, "Failed to convert URL", urise));
            }
            final IFileStore templateFileStore = EFS.getStore(templateURI) ;
            final IFileStore destinationFileStore = EFS.getStore(f_project.getLocationURI()) ;
            templateFileStore.copy(destinationFileStore, EFS.NONE, monitor) ;
	}
	
	/**
	 * 
	 * @param f_page the WizardNewProjectCreationPage
         * @param f_serverPage The ESB server page.
	 * @param f_project the IProject being created
	 * @param f_workbench workbench
	 * @param monitor monitor
	 * @throws JavaModelException java model exception
	 * @throws CoreException core exception
	 */
	public void createJavaProject(WizardNewProjectCreationPage f_page, ESBServerPage f_serverPage,
                
			IProject f_project, IWorkbench f_workbench, IProgressMonitor monitor)
			throws JavaModelException, CoreException, IOException {
			monitor.beginTask("Creating ESB Package", 0);
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			IProject project = root.getProject(f_page.getProjectName());
				
			IProjectDescription description = ResourcesPlugin.getWorkspace().newProjectDescription(project.getName());
				
			if(!Platform.getLocation().equals(f_page.getLocationPath()))
				description.setLocation(f_page.getLocationPath());
				
			description.setNatureIds(new String[] { JavaCore.NATURE_ID });
			ICommand buildCommand = description.newCommand();
			buildCommand.setBuilderName(JavaCore.BUILDER_ID);
			description.setBuildSpec(new ICommand[] { buildCommand });
			project.create(description, monitor);
			project.open(monitor);

			IJavaProject javaProject = JavaCore.create(project);
				
			IFolder srcFolder = project.getFolder("src");
			srcFolder.create(true, true, monitor);

			IFolder binFolder = project.getFolder("bin");
			binFolder.create(true, true, monitor);
				
                        copyTemplateFiles(project, monitor);
			
			javaProject.setOutputLocation(binFolder.getFullPath(), monitor);
			
                        final String esbLocation = f_serverPage.getESBLocation() ;
                        final String asLocation = f_serverPage.getASLocation() ;
                        
			final IClasspathEntry[] libraryClasspath ;
                        if (asLocation != null)
                        {
                            libraryClasspath = ESBUtils.getESBArtifactClasspath(esbLocation, asLocation) ;
                        }
                        else
                        {
                            libraryClasspath = ESBUtils.getESBServerArtifactClasspath(esbLocation) ;
                        }
			IClasspathEntry[] classpath = new IClasspathEntry[2 + libraryClasspath.length];
			// Source entry must be first in classpath
			classpath[0] = JavaCore.newSourceEntry(project.getProject().getFullPath().append(new Path("/src")));
			// Add the JRE for the classpath
			classpath[1] = JavaRuntime.getDefaultJREContainerEntry();
			// Add all of the jars
                        System.arraycopy(libraryClasspath, 0, classpath, 2, libraryClasspath.length) ;
			
			// Set the raw classpath
			javaProject.setRawClasspath(classpath, binFolder.getFullPath(), monitor);
			
                        // create build.properties file
                        final IFile propertyFile = project.getFile("build.properties") ;
                        final IPath basePath = Path.fromOSString(esbLocation) ;
                        final IPath confDir = basePath.append("samples/quickstarts/conf") ;
                        final String confDirValue = confDir.makeAbsolute().toOSString() ;
                        
                        final Properties props = new Properties() ;
                        props.put(QUICKSTART_CONF_PROPERTY, confDirValue) ;
                        final ByteArrayOutputStream baos = new ByteArrayOutputStream() ;
                        props.store(baos, "# Autogenerated by ESB Eclipse Wizard") ;
                        
                        propertyFile.create(new ByteArrayInputStream(baos.toByteArray()), false, monitor) ;
                        
			project.refreshLocal(IResource.DEPTH_INFINITE, null);
			monitor.worked(1);
	}
}
