/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2007
 */

package org.jboss.soa.esb.eclipse.wizards;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;

import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.part.ISetSelectionTarget;
import org.jboss.soa.esb.eclipse.preferences.ESBPreferenceConstants;
import org.jboss.soa.esb.eclipse.template.Activator;


/**
 * This is a new wizard which creates a very simple ESB project.    The
 * wizard copies over the requisite files, sets the classpath, and refreshes.
 * Using the default WizardNewProjectCreationPage for now because all
 * that is needed to start is a project name.     If we want to customize
 * the class names or the package names we're going to have to extend that a bit.
 */
public class ESBNewWizard extends Wizard implements INewWizard {
	private WizardNewProjectCreationPage m_page;
        private ESBServerPage m_serverPage;
	private IProject m_project;
	private IWorkbench m_workbench;
	
	/**
	 * Constructor for ESBNewWizard.
	 */
	public ESBNewWizard() {
		super();
		//If you are writing an Eclipse plug-in, you can use the services provided by org.eclipse.core.runtime.ILog 
		//that is accessed through the method getLog() from your Plugin class. 
		//Just create an instance of org.eclipse.core.runtime.Status with the right information and call the log() method on ILog.
		setNeedsProgressMonitor(true);
		Activator.getDefault().getLog().log(new Status(Status.INFO, Activator.PLUGIN_ID, Status.OK, "JRL-plugin-msg-onESBNewWizard", null));
	}
	
	/**
	 * Adding the default new project wizard page to the wizard.
	 */
	public void addPages() {
		super.addPages();
		setWindowTitle("New ESB Package Project");
                
		m_page = new WizardNewProjectCreationPage("WizardNewProjectCreationPage");
		m_page.setTitle("ESB Package Project");
		m_page.setDescription("Create a new ESB package project");
		addPage(m_page);
                
                m_serverPage = new ESBServerPage("ESBServerPage") ;
                m_serverPage.setTitle("ESB Server Location") ;
                m_serverPage.setDescription("Specify location of ESB installation and Application Server") ;
                
                final IPreferenceStore preferenceStore = Activator.getDefault().getPreferenceStore() ;
                
                final String esbLocation = preferenceStore.getString(ESBPreferenceConstants.ESB_LOCATION) ;
                final String asLocation = preferenceStore.getString(ESBPreferenceConstants.AS_LOCATION) ;
                final String asProfileLocation = preferenceStore.getString(ESBPreferenceConstants.AS_PROFILE_LOCATION);
                m_serverPage.setInitialESBLocation(esbLocation) ;
                m_serverPage.setInitialASLocation(asLocation) ;
                m_serverPage.setInitialASProfileLocation(asProfileLocation);
                addPage(m_serverPage) ;
	}

	/**
	 * Create the project, start and complete the monitor progress.
	 */
	private void createProject(IProgressMonitor monitor) throws InterruptedException, CoreException, InvocationTargetException
	{
		if (monitor == null) {
			monitor= new NullProgressMonitor();
		}
		try {		
			String strName = m_page.getProjectName();
			monitor.beginTask("Creating "+ strName + " ESB Project", 3);
	
			IProject project= m_page.getProjectHandle();
			IPath locationPath= m_page.getLocationPath();
			
			// create the project
			IProjectDescription desc= project.getWorkspace().newProjectDescription(project.getName());
			if (!m_page.useDefaults()) {
				desc.setLocation(locationPath);
			}

			ESBProjectCreator esbpc = new ESBProjectCreator();
                        try {
                            esbpc.createJavaProject(m_page, m_serverPage, project, m_workbench, monitor);
                        } catch (final IOException ioe) {
                            throw new InvocationTargetException(ioe) ;
                        }
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * performFinish is called when the user hits the "Finish" button.
	 */
	public boolean performFinish() {
                m_project = m_page.getProjectHandle();
                WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
                    protected void execute(IProgressMonitor monitor)
                        throws CoreException, InterruptedException, InvocationTargetException {
                        try {
                                createProject(monitor);
                                selectAndReveal(m_project, m_workbench.getActiveWorkbenchWindow());
                        } finally {
                                monitor.done();
                        }
                    }
                };
                try {
                        getContainer().run(false, true, op);
                } catch (InvocationTargetException ite) {
                    final Throwable targetException = ite.getTargetException() ;
                    if (targetException instanceof CoreException) {
                        ErrorDialog.openError(getShell(), "Error creating project",
                            null, ((CoreException)targetException).getStatus());
                    } else {
                        final String targetMessage = targetException.getMessage() ;
                        final String message = (targetMessage != null ? targetMessage : targetException.toString()) ;
                        final IStatus status = new Status(IStatus.ERROR, Activator.PLUGIN_ID, 0, message, targetException);
                        Activator.getDefault().getLog().log(status) ;
                        MessageDialog.openError(getShell(), "Error creating project", targetException.getMessage()) ;
                    }
                } catch (InterruptedException e) {
                        return false ;
                }
                selectAndReveal(m_project, m_workbench.getActiveWorkbenchWindow());
                return true;
	}
	
	/**
	 * Refresh the workbench.
	 * @param resource resource
	 * @param window window
	 */
    private void selectAndReveal(IResource resource,
	           IWorkbenchWindow window) {
		if (!inputValid(resource, window)) return;   
		Iterator itr = getParts(window.getActivePage()).iterator();
		while (itr.hasNext()) {
		    selectAndRevealTarget(
					window, 
					new StructuredSelection(resource), 
					getTarget((IWorkbenchPart)itr.next()));
		}
	}
	
	/**
	 * Return a target from a IWorkbenchPart.
	 * @param part workbench part
	 * @return target
	 */
    private ISetSelectionTarget getTarget(IWorkbenchPart part) {
        ISetSelectionTarget target = null;
        if (part instanceof ISetSelectionTarget) {
            target = (ISetSelectionTarget)part;
        }
        else {
            target = (ISetSelectionTarget)part.getAdapter(ISetSelectionTarget.class);
        }
		return target;		
	}
    
    /**
     * Check whether the window / resource combination is valid.
     * @param resource
     * @param window
     * @return
     */
	private boolean inputValid(IResource resource, IWorkbenchWindow window) {
		if (window == null || resource == null) return false;
		else if (window.getActivePage() == null) return false;
		else return true;
	}

	/**
	 * Call select reveal.
	 * @param window window
	 * @param selection selection
	 * @param target target
	 */private void selectAndRevealTarget(IWorkbenchWindow window, final ISelection selection, ISetSelectionTarget target) {
		if (target == null) return;
		final ISetSelectionTarget finalTarget = target;
		window.getShell().getDisplay().asyncExec(new Runnable() {
		    public void run() {
		        finalTarget.selectReveal(selection);
		    }
		});
	}
    
	 /**
	  * Return a list of parts to iterate over.
	  * @param page
	  * @return list of parts
	  */
	private List getParts(IWorkbenchPage page) {
		ArrayList result = new ArrayList();
		addParts(result, page.getViewReferences());
		addParts(result, page.getEditorReferences());
		return result;
	}
	
	/**
	 * Copy all of the workbench part references into the list.
	 * @param parts list of parts
	 * @param refs workbench part references
	 */
	private void addParts(ArrayList parts, IWorkbenchPartReference[] refs) {
		for (int i = 0; i < refs.length; i++) {
           IWorkbenchPart part = refs[i].getPart(false);
           if (part != null) {
               parts.add(part);
           }
	    }		
	}
	
	/**
	 * We will accept the selection in the workbench to see if
	 * we can initialize from it.
	 * @see IWorkbenchWizard#init(IWorkbench, IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		this.m_workbench = workbench;
		this.m_project = null;
		setNeedsProgressMonitor(true);
	}
}