/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2007
 */

package org.jboss.soa.esb.eclipse.util;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.JavaCore;

/**
 * Utility methods for identifying ESB and App Server deployments.
 * @author kevin
 */
public class ESBUtils
{
    /**
     * The name of the ESB sar.
     */
    private static final String ESB_SAR = "jbossesb.sar" ;
    /**
     * The current ESB artifacts.
     */
    private static final String[] ESB_ARTIFACTS = {
        "jbossesb.esb", "jbpm.esb", "jbrules.esb",
        "smooks.esb", "soap.esb", "spring.esb"
    } ;
    
    /**
     * Does the specified location represent an ESB Server installation with a default profile?
     * @param location The specified location.
     * @return true if it is an ESB installation, false otherwise.
     */
    public static boolean isESBServerLocation(final String location)
    {
        return isESBServerLocation(location,"default");
    }
    
    /**
     * Does the specified location represent an ESB Server installation?
     * @param location The specified location.
     * @param profile 
     * @return true if it is an ESB installation, false otherwise.
     */
    public static boolean isESBServerLocation(final String location, String profile)
    {
        // esb installation has EULA, quickstarts and lib esb files
        final String normalisedLocation = normalise(location) ;
        if (normalisedLocation != null)
        {
            final IPath path = Path.fromOSString(normalisedLocation) ;
            final IFileStore esbFileStore = EFS.getLocalFileSystem().getStore(path) ;
            return (isDirectory(esbFileStore) && hasEULA(esbFileStore) &&
                hasQuickstarts(esbFileStore) && hasDeployESBDirs(esbFileStore, profile)) ;
        }
        return false ;
    }
    
    /**
     * Does the specified location represent an ESB server or ESB deployment?
     * @param location The specified location.
     * @return true if it is an App Server or ESB deployment, false otherwise.
     */
    public static boolean isESBServerOrESBLocation(final String location)
    {
    	return isESBServerOrESBLocation(location, "default");
    }
    
    /**
     * Does the specified location represent an ESB server or ESB deployment?
     * @param location The specified location.
     * @param profileLocation 
     * @return true if it is an App Server or ESB deployment, false otherwise.
     */
    public static boolean isESBServerOrESBLocation(final String location, String profileLocation)
    {
        // esb server installation has EULA, quickstarts and server/default/deploy esb files
        final String normalisedLocation = normalise(location) ;
        if (normalisedLocation != null)
        {
            final IPath path = Path.fromOSString(normalisedLocation) ;
            final IFileStore esbFileStore = EFS.getLocalFileSystem().getStore(path) ;
            return (isDirectory(esbFileStore) && hasEULA(esbFileStore) &&
                hasQuickstarts(esbFileStore) &&
                (hasDeployESBDirs(esbFileStore, profileLocation) || hasLibESBDirs(esbFileStore))) ;
        }
        return false ;
    }
    
    /**
     * Does the specified location represent an App Server deployment?
     * @param location The specified location.
     * @return true if it is an App Server deployment, false otherwise.
     */
    public static boolean isASLocation(final String location)
    {
        // as server installation has server, client libs
        final String normalisedLocation = normalise(location) ;
        if (normalisedLocation != null)
        {
            final IPath path = Path.fromOSString(normalisedLocation) ;
            final IFileStore asFileStore = EFS.getLocalFileSystem().getStore(path) ;
            return (isDirectory(asFileStore) && hasServer(asFileStore) &&
                hasClientLibs(asFileStore)) ;
        }
        return false ;
    }
    
    /**
     * Normalise the location string to remove spaces.
     * @param location The location.
     * @return null if empty, trimmed location otherwise.
     */
    private static String normalise(final String location)
    {
        if (location != null)
        {
            final String trimmedLocation = location.trim() ;
            return (trimmedLocation.length() > 0 ? trimmedLocation : null) ;
        }
        return null ;
    }
    
    /**
     * Check the EULA, it should exist in all ESB installations.
     * @param location The base location to check.
     * @return true if the EULA exists, false otherwise.
     */
    private static boolean hasEULA(final IFileStore location)
    {
        final IFileStore eulaFileStore = location.getChild("JBossORG-EULA.txt") ;
        final IFileInfo eulaFileInfo = eulaFileStore.fetchInfo() ;
        return (eulaFileInfo.exists() && !eulaFileInfo.isDirectory()) ; 
    }
    
    /**
     * Check the quickstarts, they should exist in all ESB installations.
     * @param location The base location to check.
     * @return true if the EULA exists, false otherwise.
     */
    private static boolean hasQuickstarts(final IFileStore location)
    {
        final IFileStore samples = location.getChild("samples") ;
        if (isDirectory(samples))
        {
            final IFileStore quickstarts = samples.getChild("quickstarts") ;
            return isDirectory(quickstarts) ;
        }
        return false ;
    }
    
    /**
     * Are the ESB artifacts in the lib directory?
     * @param location The base location to check.
     * @return true if the ESB artifacts are in lib, false otherwise.
     */
    private static boolean hasLibESBDirs(final IFileStore location)
    {
        final IFileStore libDir = location.getChild("lib") ;
        return (isDirectory(libDir) && hasESBArtifacts(libDir)) ;
    }
    
    /**
     * Are the ESB artifacts in the "default" deploy directory?
     * @param location The base location to check.
     * @return true if the ESB artifacts are in deploy, false otherwise.
     */
    private static boolean hasDeployESBDirs(final IFileStore location)
    {
    	return hasDeployESBDirs(location,"default");
    }
    
    /**
     * Are the ESB artifacts in the deploy directory?
     * @param location The base location to check.
     * @param profile 
     * @return true if the ESB artifacts are in deploy, false otherwise.
     */
    private static boolean hasDeployESBDirs(final IFileStore location, String profile)
    {
    	final IFileStore serverDir = location.getChild("server") ;
    	if (isDirectory(serverDir))
    	{
        	final IFileStore defaultDir = serverDir.getChild(profile) ;
            if (isDirectory(defaultDir))
            {
                final IFileStore deployDir = defaultDir.getChild("deploy") ;
                return (isDirectory(deployDir) && hasESBArtifacts(deployDir)) ;
            }
        }
        return false ;
    }
    
    /**
     * Does the location contain the app server client libs?
     * @param location The base location to check.
     * @return true if the AS client libs are present, false otherwise.
     */
    private static boolean hasClientLibs(final IFileStore location)
    {
        final IFileStore client = location.getChild("client") ;
        if (isDirectory(client))
        {
            final IFileStore jbossAll = client.getChild("jbossall-client.jar") ;
            return jbossAll.fetchInfo().exists() ;
        }
        return false ;
    }
    
    /**
     * Does the location contain the AS directory?
     * @param location The base location to check.
     * @return true if the AS server directory is present, false otherwise.
     */
    private static boolean hasServer(final IFileStore location)
    {
        final IFileStore server = location.getChild("server") ;
        return isDirectory(server) ;
    }
    
    /**
     * Does the location contain the ESB artifacts>
     * @param location The base location to check.
     * @return true if the artifacts are present, false otherwise.
     */
    private static boolean hasESBArtifacts(final IFileStore location)
    {
        final int numArtifacts = ESB_ARTIFACTS.length ;
        for(int count = 0 ; count < numArtifacts ; count++)
        {
            final String artifact = ESB_ARTIFACTS[count] ;
            final IFileStore artifactFile = location.getChild(artifact) ;
            if (!artifactFile.fetchInfo().exists())
            {
                return false ;
            }
        }
        
        final IFileStore esbSarFile = location.getChild(ESB_SAR) ;
        return esbSarFile.fetchInfo().exists() ;
    }
    
    /**
     * Does the specified file represent a directory?
     * @param file The file.
     * @return true if it is a directory which exists, false otherwise.
     */
    private static boolean isDirectory(final IFileStore file)
    {
        final IFileInfo fileInfo = file.fetchInfo() ;
        return (fileInfo.exists() && fileInfo.isDirectory()) ;
    }

    /**
     * Get the artifact classpath for the ESB server using default profile.
     * @param esbLocation The ESB location.
     * @return The classpath entries for the artifact classpath.
     * @throws CoreException 
     * @throws CoreException for errors.
     */
    public static IClasspathEntry[] getESBServerArtifactClasspath(final String esbLocation) throws CoreException
    {
    	return getESBServerArtifactClasspath(esbLocation,"default");
    }
    
    /**
     * Get the artifact classpath for the ESB server.
     * @param esbLocation The ESB location.
     * @param asProfileLocation The AS Profile location.
     * @return The classpath entries for the artifact classpath.
     * @throws CoreException for errors.
     */
    public static IClasspathEntry[] getESBServerArtifactClasspath(final String esbLocation, final String asProfileLocation)
        throws CoreException
    {
        final ArrayList classpathEntries = new ArrayList() ;
        final HashSet jarEntries = new HashSet() ;
        final IPath esbLocationPath = Path.fromOSString(esbLocation) ;
        final IPath deployPath = esbLocationPath.append("server/" + asProfileLocation + "/deploy") ;
        
        addSARArtifactJars(classpathEntries, jarEntries, deployPath, ESB_SAR) ;
        addESBArtifactJars(classpathEntries, jarEntries, deployPath, ESB_ARTIFACTS) ;
        addClientJars(classpathEntries, jarEntries, esbLocationPath) ;
        
        return (IClasspathEntry[]) classpathEntries.toArray(new IClasspathEntry[classpathEntries.size()]) ;
    }
    
    /**
     * Get the artifact classpath for the ESB server.
     * @param esbLocation The ESB location.
     * @param asLocation The AS location.
     * @return The classpath entries for the artifact classpath.
     * @throws CoreException for errors.
     */
    public static IClasspathEntry[] getESBArtifactClasspath(final String esbLocation, final String asLocation)
        throws CoreException
    {
        final ArrayList classpathEntries = new ArrayList() ;
        final HashSet jarEntries = new HashSet() ;
        final IPath esbLocationPath = Path.fromOSString(esbLocation) ;
        final IPath asLocationPath = Path.fromOSString(asLocation) ;
        
        final IPath esbLibPath = esbLocationPath.append("lib") ;
        addSARArtifactJars(classpathEntries, jarEntries, esbLibPath, ESB_SAR) ;
        addESBArtifactJars(classpathEntries, jarEntries, esbLibPath, ESB_ARTIFACTS) ;
        addClientJars(classpathEntries, jarEntries, asLocationPath) ;
        
        return (IClasspathEntry[]) classpathEntries.toArray(new IClasspathEntry[classpathEntries.size()]) ;
    }
    
    /**
     * Add the SAR artifact jars to the classpath.
     * @param classpathEntries The current classpath list.
     * @param jarEntries The set of jars already added.
     * @param parent The parent of the sar artifact.
     * @param sarArtifact The name of the sar artifact.
     * @throws CoreException for errors.
     */
    private static void addSARArtifactJars(final List classpathEntries, final Set jarEntries, final IPath parent, final String sarArtifact)
        throws CoreException
    {
        final IPath sarLibPath = parent.append(sarArtifact + "/lib") ;
        addAllJars(classpathEntries, jarEntries, sarLibPath) ;
    }
    
    /**
     * Add the ESB artifact jars to the classpath.
     * @param classpathEntries The current classpath list.
     * @param jarEntries The set of jars already added.
     * @param parent The parent of the ESB artifacts.
     * @param esbArtifacts The names of the esb artifacts.
     * @throws CoreException for errors.
     */
    private static void addESBArtifactJars(final List classpathEntries, final Set jarEntries, final IPath parent, final String[] esbArtifacts)
        throws CoreException
    {
        final int numArtifacts = (esbArtifacts == null ? 0 : esbArtifacts.length) ;
        for(int count = 0 ; count < numArtifacts ; count++)
        {
            final IPath artifactPath = parent.append(esbArtifacts[count]) ;
            addAllJars(classpathEntries, jarEntries, artifactPath) ;
        }
    }
    
    /**
     * Add the App Server client jars to the classpath.
     * @param classpathEntries The current classpath list.
     * @param jarEntries The set of jars already added.
     * @param asRoot The root of the app server install.
     */
    private static void addClientJars(final List classpathEntries, final Set jarEntries, final IPath asRoot)
    {
        final IPath clientJarPath = asRoot.append("client/jbossall-client.jar") ;
        addEntry(classpathEntries, jarEntries, clientJarPath) ;
    }
    
    /**
     * Add all jars in the specified directory.
     * @param classpathEntries The current classpath list.
     * @param jarEntries The set of jars already added.
     * @param directory The directory containing the jars.
     * @throws CoreException for errors.
     */
    private static void addAllJars(final List classpathEntries, final Set jarEntries, final IPath root)
        throws CoreException
    {
        final IFileStore rootStore = EFS.getLocalFileSystem().getStore(root) ;
        final String[] names = rootStore.childNames(EFS.NONE, null);
        final int numNames = (names == null ? 0 : names.length) ;
        for(int count = 0 ; count < numNames ; count++)
        {
            final String child = names[count] ;
            if ((child != null) && child.endsWith(".jar"))
            {
                final IFileStore childStore = rootStore.getChild(child) ;
                final IFileInfo childInfo = childStore.fetchInfo() ;
                if (childInfo.exists() && !childInfo.isDirectory())
                {
                    final IPath childPath = root.append(child) ;
                    addEntry(classpathEntries, jarEntries, childPath) ;
                }
            }
        }
    }
    
    /**
     * Add the jar entry to the classpath, checking for duplicates.
     * @param classpathEntries The current classpath list.
     * @param jarEntries The set of jars already added.
     * @param jarPath The current jar to add.
     */
    private static void addEntry(final List classpathEntries, final Set jarEntries, final IPath jarPath)
    {
        final String jarName = jarPath.lastSegment() ;
        if (jarEntries.add(jarName))
        {
            classpathEntries.add(JavaCore.newLibraryEntry(jarPath, null, null)) ;
        }
    }

	public static String[] getValidProfiles(final String location) {
		final String normalisedLocation = normalise(location) ;
        if (normalisedLocation != null){
            final IPath path = Path.fromOSString(normalisedLocation) ;
            final IFileStore serverDir = EFS.getLocalFileSystem().getStore(path) ;
            try {
				return serverDir.getChild("server").childNames(EFS.NONE, null);
			} catch (CoreException e) {
				e.printStackTrace();
			}
         }	
			
		return null;
	}
}
