/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.transform;

import static org.jboss.seam.ScopeType.CONVERSATION;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import javax.persistence.EntityManager;

import org.jboss.seam.annotations.Begin;
import org.jboss.seam.annotations.End;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Out;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.datamodel.DataModel;
import org.jboss.seam.annotations.datamodel.DataModelSelectionIndex;
import org.jboss.soa.esb.admin.console.SeamViewIdStack;
import org.jboss.soa.esb.admin.console.Toggle;
import org.jboss.soa.esb.admin.console.UpdateNotifier;
import org.jboss.soa.esb.admin.console.exchange.MessageExchange;
import org.jboss.soa.esb.admin.console.exchange.SelectMessageExchangeAction;

/**
 * Manage the <a href="http://milyn.codehaus.org/Smooks">Smooks</a> resource configurations targeted at the session selected message exchange.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Name("manageResources")
@Scope(CONVERSATION)
public class ManageResources implements Serializable {

	private static final long serialVersionUID = 1L;

	@In(create=true)
	private EntityManager entityManager;

	@In(required=false)
	private MessageExchange messageExchange;

	@In(create=true)
	private UpdateNotifier updateNotifier;
	
	@DataModel 
	private List<TransformationResource> messageExchangeConfigs;
	@DataModelSelectionIndex("messageExchangeConfigs")
	int messageExchangeConfigIndex = 0;
	
	@In(required=false) @Out(required=false,scope=CONVERSATION)
	private Toggle showAll;
	
	@In(required=false) @Out(required=false,scope=CONVERSATION)
	private TransformationResource selectedConfig;
	
	@DataModel 
	private List<TransformationResourceParameter> selectedConfigParams;
	@DataModelSelectionIndex("selectedConfigParams")
	int selectedConfigParamsIndex = 0;
	
	@In(required=false) @Out(required=false,scope=CONVERSATION)
	private TransformationResourceParameter selectedConfigParam;
	
	@Begin(join=true)
	public String list() {
		if(!SelectMessageExchangeAction.isMessageExchangeSelected()) {
			SeamViewIdStack.getInstance().pushViewId("list-resources");
			return "select-message-exchange";
		}

		initialise();
		
		return "list-resources";
	}
	
	public void toggleShowAll() {
		showAll.toggle();
		initialise();
	}

	public String viewConfiguration() {
		selectedConfig = messageExchangeConfigs.get(messageExchangeConfigIndex);
		selectedConfigParams = selectedConfig.getParameters();
				
		return "view-resource";
	}

	public String selectUpdateParameter() {
		selectedConfigParam = selectedConfigParams.get(selectedConfigParamsIndex);
				
		return "update-resource-parameter";
	}

	public String updateParameter() {
		entityManager.persist(selectedConfigParam);
		updateNotifier.sendNotification();
				
		return "view-resource";
	}
	
	public String deleteConfiguration() {
		selectedConfig = messageExchangeConfigs.get(messageExchangeConfigIndex);
		for(TransformationResourceParameter parameter : selectedConfig.getParameters()) {
			entityManager.remove(parameter);
		}
		entityManager.remove(selectedConfig);
		initialise();
		updateNotifier.sendNotification();
		
		return "list-resources";
	}

	@End
	public String home() {
		return "home";
	}
	
	/**
	 * Initialise the message exchange configurations list.
	 */
	@Begin(join=true)
	public void initialise() {
		List<TransformationResource> allResourceConfigs;
		
		if(showAll == null) {
			showAll = new Toggle();
		}

		// Read all resources configs from the DB...
		allResourceConfigs = entityManager.createQuery("from TransformationResource").getResultList();
		
		// Filter and sort the resource configs for the selected message exchange (for display)...
		messageExchangeConfigs = SmooksUtils.getMessageExchangeConfigs(messageExchange, allResourceConfigs);
		if(!showAll.isOn()) {
			Iterator<TransformationResource> listIterator = messageExchangeConfigs.iterator();
			while(listIterator.hasNext()) {
				TransformationResource resource = listIterator.next();
				if(!resource.getUseragent().equalsIgnoreCase(messageExchange.getExchangeUseragentString())) {
					listIterator.remove();
				}
			}
		}
		
		// Clear the selected configuration.
		selectedConfig = null;
	}
}
