/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.contract;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;

import org.hibernate.annotations.GenericGenerator;
import org.hibernate.validator.NotNull;
import org.jboss.seam.annotations.Name;

/**
 * Message contract definition.
 * <p/>
 * This class defines message properties with respect to the type of data contained in the message, as well as the
 * {@link org.jboss.soa.esb.admin.console.exchange.participant.Participant} that produces or consumes the message.  It basically
 * puts some typing information around the messages that exchange through the ESB.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Entity
@Name("messageContract")
public class MessageContract implements Serializable {

	private static final long serialVersionUID = 1L;

	/**
	 * Enumeration type for how a message type is used by the service located at the associated Participant.
	 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
	 */
	public static enum USAGE {
		PRODUCES, 
		CONSUMES
	};
	
	/**
	 * Persistence ID ref.
	 */
	private String id;
	/**
	 * The Participant that this message type is associated with.
	 */
	private String participant;
	/**
	 * How this message is used by the Participant.
	 */
	private String usage;
	/**
	 * Message Mime Type.
	 */
	private String mime;
	/**
	 * Message namespace.
	 */
	private String namespace;
	/**
	 * Message description (dtd, xsd etc).
	 */
	private String messageDescription;
	
	// TODO:  The type could/should be normalised further by pulling the message typing info into a type of its own,
	// making this class look like...
	// class MessageContract {
	//		MessageType messageType;
	// 		Participant participant;
	//		USAGE messageUsage;
	// }
	
	/**
	 * Public default constructor.
	 */
	public MessageContract() {
	}

	/**
	 * Public Constructor.
	 * @param mime Message mime type (required).
	 * @param namespace Message namespace (optional).
	 * @param messageDescription Message dtd/schema (optional).
	 */
	public MessageContract(String mime, String namespace, String messageDescription) {
		this.mime = mime;
		this.namespace = namespace;
		this.messageDescription = messageDescription;
	}

	/**
	 * @return Returns the participant.
	 */
	@NotNull
	public String getParticipant() {
		return participant;
	}

	/**
	 * @param participant The participant to set.
	 */
	public void setParticipant(String participant) {
		this.participant = participant;
	}

	/**
	 * @return Returns the usage.
	 */
	@NotNull
	public String getUsage() {
		return usage;
	}

	/**
	 * @param usage The usage to set.
	 */
	public void setUsage(String usage) {
		// TODO:  There must be a way to get Seam to perform tis checking via annotations!
		if(!USAGE.PRODUCES.name().equals(usage) && !USAGE.CONSUMES.name().equals(usage)) {
			// TODO: Handle this properly!!  Better still - find out if and how to get enums to work with Seam.
			throw new RuntimeException("Message Usage field must be either '" + USAGE.PRODUCES.name() + "' '" + USAGE.CONSUMES.name() + "'");
		}
		this.usage = usage;
	}
	
	/**
	 * @return Returns the messageDescription.
	 */
	public String getMessageDescription() {
		return messageDescription;
	}
	/**
	 * @param messageDescription The messageDescription to set.
	 */
	public void setMessageDescription(String messageDescription) {
		this.messageDescription = messageDescription;
	}
	/**
	 * @return Returns the mime.
	 */
	public String getMime() {
		return mime;
	}
	/**
	 * @param mime The mime to set.
	 */
	public void setMime(String mime) {
		this.mime = mime;
	}

	/**
	 * @return Returns the id.
	 */
	@Id @GeneratedValue(generator="system-uuid")
	@GenericGenerator(name="system-uuid", strategy = "uuid")
	public String getId() {
		return id;
	}

	/**
	 * @param id The id to set.
	 */
	public void setId(String id) {
		this.id = id;
	}

	/**
	 * @return Returns the message namespace.
	 */
	@NotNull
	public String getNamespace() {
		return namespace;
	}
	
	/**
	 * @param namespace The message namespace to set.
	 */
	public void setNamespace(String xmlns) {
		this.namespace = xmlns;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuffer stringBuf = new StringBuffer();
		
		stringBuf.append(mime);
		if(namespace != null) {
			if(mime != null && !mime.trim().equals("")) {
				stringBuf.append(':');
			}
			stringBuf.append(namespace);
		}
		
		return stringBuf.toString();
	}

	/**
	 * Map the suppied list of Message Contracts to a list of contract DTO instances.
	 * @param contracts The list of contracts to be mapped.
	 * @return The list of mapped DTO instances.
	 */
	public static List<MessageContractDTO> toDTOList(List<MessageContract> contracts) {
		List<MessageContractDTO> list = new ArrayList<MessageContractDTO>();
		
		for(MessageContract contract : contracts) {
			list.add(new MessageContractDTO(contract.id, contract.toString()));
		}
		
		return list;
	}
}
