package org.jboss.soa.esb.monitoring.server;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

import java.util.List;

import org.apache.log4j.Logger;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.message.body.content.ServiceControlCommand;
import org.jboss.soa.esb.monitoring.MonitoringSessionFactory;
import org.jboss.soa.esb.monitoring.OperationsData;
import org.jboss.soa.esb.monitoring.pojo.JMXOperation;

/**
 * OperationsFiler converts the data from an OperationsData object to a JMXOperation
 * and then persists the data.
 * 
 * @author <a href="mailto:tcunning@redhat.com">Tom Cunningham</a>
 * @since Version 4.2
 */
public class OperationsFiler implements Filer {
	private OperationsData data;
	private static final Logger logger = Logger.getLogger(OperationsFiler.class);
	
	/**
	 * Null argument constructor.
	 */
	public OperationsFiler() {
	}
	
	/**
	 * Constructor
	 * @param data operations data
	 */
	public OperationsFiler(OperationsData data) {
		this.data = data;
	}
	
	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.monitoring.server.Filer#setData(java.lang.Object)
	 */
	public void setData(Object data) {
		this.data = (OperationsData) data;
	}
	
	/**
	 * @param sess session 
	 * @param objectname object name
	 * @param attribute attribute name
	 * @return JMXAttribute
	 */
	public JMXOperation getOperation(Session sess, String objectname, String operation) {
		String query = "from JMXOperation jmxo "
					+ "where objectname = :objectname and operation = :operation";
				
		List result = sess.createQuery(query)
			.setString("objectname", objectname)
			.setString("operation", operation).list();
		
		if (result.size() > 0) {
			JMXOperation jmxo = (JMXOperation) result.get(0);
			return jmxo;
		}
	 	return null;
	}
		
	/**
	 * Create a JMXAttribute object based on the StatisticBean info and stores it.
	 * @param f_sb statistics bean
	 */
	public void insertOperations(ServiceControlCommand f_ob) {
		Session sess = null;
		Transaction tx = null;
		TransactionStrategy txS = TransactionStrategy.getTransactionStrategy(true);
		Object txHandle = null;
		
		if (txS != null)
		{
			try
			{
				txHandle = txS.suspend();
			}
			catch (TransactionStrategyException ex)
			{
			}
		}
		
		try {
			sess = (Session) MonitoringSessionFactory.getInstance().openSession();
			tx = sess.beginTransaction();

			JMXOperation oper = getOperation(sess, f_ob.getObjectName(), f_ob.getOperation());
			if (oper == null) {
				oper = new JMXOperation(f_ob.getObjectName(), f_ob.getServer(), f_ob.getOperation(),
						f_ob.getDescription(), f_ob.getReturntype(), true);
				sess.save(oper);
			} else {
				oper.setActiveflag(new Boolean(true));
				sess.save(oper);
			}
			tx.commit();

		} catch (Exception e) {
			logger.error("", e);
		} finally {
			if (tx.isActive()) {
				tx.rollback();
			} 
			tx = null;
			
			sess.flush();
			sess.close();
			sess = null;
			
			if (txHandle != null)
			{
				try
				{
					txS.resume(txHandle);
				}
				catch (TransactionStrategyException ex)
				{
					logger.error("Problem resuming transaction!", ex);
				}
			}
		}
	}
	
	/**
	 * The JMXOperation table contains a flag which represents whether the operation is 
	 * valid - has the resource been undeployed.
	 * @param serverName server name
	 */
	public void updateActiveFlag(String serverName) {
		Session sess = null;
		Transaction tx = null;
		TransactionStrategy txS = TransactionStrategy.getTransactionStrategy(true);
		Object txHandle = null;
		
		if (txS != null)
		{
			try
			{
				txHandle = txS.suspend();
			}
			catch (TransactionStrategyException ex)
			{
			}
		}
		
		try {
			sess = (Session) MonitoringSessionFactory.getInstance().openSession();
			tx = sess.beginTransaction();

			String queryString= "update JMXOperation jo set jo.activeflag = false "
					+ "where jo.servername = :server";
			Query query = sess.createQuery(queryString);
			query.setString("server", serverName);
			query.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			logger.error("", e);
		} finally {
			if (tx.isActive()) {
				tx.rollback();
			} 
			tx = null;
			
			sess.flush();
			sess.close();
			sess = null;
			
			if (txHandle != null)
			{
				try
				{
					txS.resume(txHandle);
				}
				catch (TransactionStrategyException ex)
				{
					logger.error("Problem resuming transaction!", ex);
				}
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.monitoring.server.Filer#persistData()
	 */
	public void persistData() {
		if (data != null) {
			List list = data.getList();
			for (int i = 0; i < list.size(); i++) {
				ServiceControlCommand bean = (ServiceControlCommand) list.get(i);
				if ((i == 0) && (bean != null)) {
					updateActiveFlag(bean.getServer());
				}
				insertOperations(bean);
			}
		}
	}
}
