/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.monitoring.server;

import java.util.List;

import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.monitoring.MonitoringSessionFactory;
import org.jboss.soa.esb.monitoring.StatisticsBean;
import org.jboss.soa.esb.monitoring.StatisticsData;
import org.jboss.soa.esb.monitoring.pojo.*;

/**
 * DataFiler receives StatisticsData information and stores it in 
 * the database using Hibernate.
 * 
 * @author <a href="mailto:tcunning@redhat.com">Tom Cunningham</a>
 * @since Version 4.2
 */
public class DataFiler implements Filer {
	private StatisticsData data;
	private static final Logger logger = Logger.getLogger(DataFiler.class);
	
	/**
	 * Null constructor.
	 */
	public DataFiler() {
	}
	
	/**
	 * Constructor taking a StatisticsData object.
	 * @param f_data
	 */
	public DataFiler(StatisticsData f_data) {
		data = f_data;
	}
	
	/**
	 * @param f_data
	 */
	public void setData(Object f_data) {
		data = (StatisticsData) f_data;
	}
	
	/**
	 * @param sess session 
	 * @param objectname object name
	 * @param attribute attribute name
	 * @return JMXAttribute
	 */
	public JMXAttribute getAttribute(Session sess, String objectname, String attribute) {
		String query = "from JMXAttribute jmxa "
					+ "where objectname = :objectname and attribute = :attribute";
				
		List result = sess.createQuery(query)
			.setString("objectname", objectname)
			.setString("attribute", attribute).list();
		
		if (result.size() > 0) {
			JMXAttribute jmxa = (JMXAttribute) result.get(0);
			return jmxa;
		}
		return null;
	}

	/**
	 * Inserts data through hibernate.
	 * @param sess session
	 * @param f_sb StatisticsBean
	 * @param attr JMXAttribute
	 */
	public void insertData(Session sess, StatisticsBean f_sb, JMXAttribute attr) {
		JMXData jmxd = null;
		if ( (attr.getAttribute().matches("(?i).*Count")) 
			|| (attr.getAttribute().matches("(?i).*Bytes")) ){
			if (f_sb.getType().equals("java.lang.Integer")) {
				Integer data = new Integer(0);
				try {
					data = (Integer) f_sb.getData();
				} catch (Exception e) {
				}
				jmxd = new JMXData(null, f_sb.getServer(), f_sb.getCollectionTime(), null, data, null);
			} else if (f_sb.getType().equals("long")) {
				Long data = (Long) f_sb.getData();
				jmxd = new JMXData(null, f_sb.getServer(), f_sb.getCollectionTime(), null, new Integer(data.intValue()), null);
			} else if (f_sb.getType().equals("int")) {
				Integer data = (Integer) f_sb.getData();
				jmxd = new JMXData(null, f_sb.getServer(), f_sb.getCollectionTime(), null, data, null);
			} else {
				logger.error ("ERROR - found type of " + f_sb.getType() + " for " + f_sb.getAttribute());
				return;
			} 
		} else if (attr.getAttribute().matches("(?i).*Time")) {
			if (f_sb.getType().equals("java.lang.Double")) {
				Double tmp = (Double) f_sb.getData();
				Float data = new Float(tmp.floatValue());
				jmxd = new JMXData(null, f_sb.getServer(), f_sb.getCollectionTime(), data, null, null);
			} else if (f_sb.getType().equals("java.lang.Float")) {
				Float data = (Float) f_sb.getData();
				jmxd = new JMXData(null, f_sb.getServer(), f_sb.getCollectionTime(), data, null, null);
			} else if (f_sb.getType().equals("java.lang.Long")) {
				Long temp = (Long) f_sb.getData();
				Float data = new Float(temp.floatValue());
				jmxd = new JMXData(null, f_sb.getServer(), f_sb.getCollectionTime(), data, null, null);
			} else if (f_sb.getType().equals("java.lang.Integer")) {
				Integer temp = (Integer) f_sb.getData();
				Float data = new Float(temp.floatValue());
				jmxd = new JMXData(null, f_sb.getServer(), f_sb.getCollectionTime(), data, null, null);				
			} else {
				logger.error ("ERROR - found type of " + f_sb.getType() + " for " + f_sb.getAttribute());
				return;
			} 
		} else if (attr.getAttribute().matches("State")) {
			if (f_sb.getType().equals("int")) {
				Integer tmp = (Integer) f_sb.getData();
				jmxd = new JMXData(null, f_sb.getServer(), f_sb.getCollectionTime(), null, null, tmp.toString());			
			}
		} else {
			if (f_sb.getType().equals("java.lang.String")) {
				String data = (String) f_sb.getData();
				jmxd = new JMXData(null, f_sb.getServer(), f_sb.getCollectionTime(), null, null, data);			
			} else {
				logger.error ("ERROR - found type of " + f_sb.getType() + " for " + f_sb.getAttribute());
				return;
			} 
		}
		

		jmxd.setAttribute(attr);
		try {
			sess.save(jmxd);
		} catch (Exception e) {
			logger.error("Problem saving " + jmxd.toString(), e);
		}
	}
		
	/**
	 * Create a JMXAttribute object based on the StatisticBean info and stores it.
	 * @param f_sb statistics bean
	 */
	public void insertStatistics(StatisticsBean f_sb) {
		TransactionStrategy txStrategy = TransactionStrategy.getTransactionStrategy(true);
		Object txHandle = null;
		
		if (txStrategy != null)
		{
			try
			{
				txHandle = txStrategy.suspend();
			}
			catch (TransactionStrategyException ex)
			{
				// if it failed, then the next work will fail too.
			}
		}
		
		Session sess = null;
		Transaction tx = null;
		try {
			sess = (Session) MonitoringSessionFactory.getInstance().openSession();
			tx = sess.beginTransaction();

			JMXAttribute attr = getAttribute(sess, f_sb.getObjectName(), f_sb.getAttribute());
			if (attr == null) {
				attr = new JMXAttribute(f_sb.getObjectName(), f_sb.getAttribute());
				sess.save(attr);
				insertData(sess, f_sb, attr);
			} else {
				insertData(sess, f_sb, attr);
			}
			tx.commit();

		} catch (Exception e) {
			logger.error("", e);
		} finally {
			if (tx.isActive()) {
				tx.rollback();
			} 
			tx = null;
			
			sess.flush();
			sess.close();
			sess = null;
			
			if (txHandle != null)
			{
				try
				{
					txStrategy.resume(txHandle);
				}
				catch (TransactionStrategyException ex)
				{
					logger.error("Problem resuming transaction!", ex);
				}
			}
		}
	}
	
	/**
	 * Calls insertStatistics to insert each StatisticBean in the StatisticData object.
	 */
	public void persistData() {
		if (data != null) {
			List list = data.getList();
			for (int i = 0; i < list.size(); i++) {
				StatisticsBean bean = (StatisticsBean) list.get(i);
				insertStatistics(bean);
			}
		}
	}
}
