/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.services.soapui;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.*;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.dom.DOMResult;

import org.apache.log4j.Logger;
import org.apache.commons.httpclient.HttpClient;
import org.jboss.internal.soa.esb.soap.OGNLUtils;
import org.jboss.internal.soa.esb.util.LRUCache;
import org.jboss.internal.soa.esb.util.ESBProperties;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.http.HttpClientFactory;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.system.ServiceMBeanSupport;
import org.milyn.xml.XmlUtil;
import org.milyn.Smooks;
import org.milyn.resource.URIResourceLocator;
import org.w3c.dom.*;
import org.xml.sax.SAXException;

import com.eviware.soapui.impl.wsdl.WsdlInterface;
import com.eviware.soapui.impl.wsdl.WsdlProject;
import com.eviware.soapui.impl.wsdl.support.wsdl.WsdlLoader;
import com.eviware.soapui.model.iface.Operation;

/**
 * Soap UI Soap Client Service MBean.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SoapUIClientService extends ServiceMBeanSupport implements SoapUIClientServiceMBean {

    private static final String SOAPUI_CLONE_COMMENT = " repetitions:";

    private static final String IS_CLONE_ATTRIB = "is-clone";
    private static Logger logger = Logger.getLogger(SoapUIClientService.class);
    private Map<String, WsdlInterface[]> wsdls = new HashMap<String, WsdlInterface[]>();
    private DocumentBuilderFactory docBuilderFactory ;
    private Map<String, Smooks> smooksCache;
    private ESBProperties properties;
    private static final String CLONED_POSTFIX = " - cloned";

    /**
     * Public default constructor.
     */
    public SoapUIClientService() throws ConfigurationException {
        properties = new ESBProperties("/soapui-client.sar.properties");

        docBuilderFactory = DocumentBuilderFactory.newInstance();
        docBuilderFactory.setNamespaceAware(true);
        int smooksLRUCacheSize = properties.getIntProperty("smooks.lru.cache.size", 30);
        smooksCache = new LRUCache<String, Smooks>(smooksLRUCacheSize);
    }

    /**
     * Build a SOAP request for the specified operation on the specified WSDL.
     *
     * @param wsdl            WSDL URL.
     * @param operation       Operation name.
     * @param params          Message parameter map.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @param smooksResource  {@link org.milyn.Smooks} transformation configuration resource.  This is the actual
     *                        Smooks resource configuration XML, not a file name.
     *                        Null if no transformations are to be performed on the SOAP message before serializing it
     *                        for return.
     * @param soapNs 		  optional SOAP namespace
     * @return The SOAP Message.
     * @throws IOException Failed to load WSDL.
     */
    public String buildRequest(String wsdl, String operation, Map params, Properties httpClientProps, String smooksResource, String soapNs) throws IOException, UnsupportedOperationException, SAXException {
        Operation operationInst = getOperation(wsdl, operation, httpClientProps);
        String requestTemplate = operationInst.getRequestAt(0).getRequestContent();

        return buildRequest(requestTemplate, params, smooksResource, soapNs);
    }

    /**
     * Get the 1st endpoint from the specified WSDL.
     *
     * @param wsdl            WSDL URL.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @return The operation endpoint URL.
     * @throws IOException Failed to load WSDL.
     */
    public String getEndpoint(String wsdl, Properties httpClientProps) throws IOException {
        WsdlInterface[] wsdlInterfaces = getWsdlInterfaces(wsdl, httpClientProps);

        return wsdlInterfaces[0].getEndpoints()[0];
    }

    private WsdlInterface[] getWsdlInterfaces(String wsdl, Properties httpClientProps) throws IOException {
        try {
            WsdlInterface[] wsdlInterfaces = wsdls.get(wsdl);
            if (wsdlInterfaces == null) {
                WsdlProject wsdlProject = new WsdlProject();
                wsdlInterfaces = wsdlProject.importWsdl(wsdl, true, createWsdlLoader(wsdl, httpClientProps));
                wsdls.put(wsdl, wsdlInterfaces);
            }
            return wsdlInterfaces;
        } catch (Exception e) {
            IOException ioe = new IOException("Failed to import WSDL '" + wsdl + "'.");
            ioe.initCause(e);
            throw ioe;
        }
    }

    private Operation getOperation(String wsdl, String operation, Properties httpClientProps) throws IOException, UnsupportedOperationException {
        WsdlInterface[] wsdlInterfaces = getWsdlInterfaces(wsdl, httpClientProps);

        for (WsdlInterface wsdlInterface : wsdlInterfaces) {
            Operation operationInst = wsdlInterface.getOperationByName(operation);

            if (operationInst != null) {
                return operationInst;
            }
        }
        
        // Try clearing WSDL cache, WSDL may have updated 
        wsdls.remove(wsdl);
        wsdlInterfaces = getWsdlInterfaces(wsdl, httpClientProps);

        for (WsdlInterface wsdlInterface : wsdlInterfaces) {
            Operation operationInst = wsdlInterface.getOperationByName(operation);

            if (operationInst != null) {
                return operationInst;
            }
        }
        
        throw new UnsupportedOperationException("Operation '" + operation + "' not supported by WSDL '" + wsdl + "'.");
    }

    private WsdlLoader createWsdlLoader(String wsdl, Properties httpClientProps) throws ConfigurationException {
        HttpClient httpClient = HttpClientFactory.createHttpClient(httpClientProps);

        return new EsbWsdlLoader(wsdl, httpClient);
    }

    private String buildRequest(String soapMessage, Map params, String smooksResource, String soapNs) throws IOException, SAXException {
        Document messageDoc = getDocBuilder().parse(new ByteArrayInputStream(soapMessage.getBytes()));

        Element docRoot = messageDoc.getDocumentElement();

        expandMessage(docRoot, params);
        injectParameters(docRoot, params, soapNs);
        if(smooksResource != null) {
            applySmooksTransform(smooksResource, messageDoc);
        }

        return XmlUtil.serialize(messageDoc.getChildNodes());
    }

    private void applySmooksTransform(String smooksResource, Document messageDoc) throws IOException, SAXException {
        if(smooksResource != null) {
            Smooks smooks = smooksCache.get(smooksResource);

            if(smooks == null) {
                smooks = new Smooks();
                smooks.addConfigurations("smooks-resource", new ByteArrayInputStream(smooksResource.getBytes("UTF-8")));
                smooks.addConfigurations("cdu-creators", new URIResourceLocator().getResource("/META-INF/smooks-creators.xml"));
                smooksCache.put(smooksResource, smooks);
            }

            smooks.filter(new DOMSource(messageDoc), new DOMResult(), smooks.createExecutionContext());        
        }
    }

    /**
     * Expand the message to accommodate data collections.
     * <p/>
     * It basically just clones the message where appropriate.
     *
     * @param element The element to be processed.
     * @param params  The message params.  Uses the message params to
     *                decide whether or not cloning is required.
     */
    private void expandMessage(Element element, Map params) {

        // If this element is not a cloned element, check does it need to be cloned...
        if (!element.hasAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, IS_CLONE_ATTRIB)) {
            String ognl = OGNLUtils.getOGNLExpression(element);
            Element clonePoint = getClonePoint(element);

            if(clonePoint != null) {
                int collectionSize;

                collectionSize = calculateCollectionSize(ognl, params);

                if(collectionSize == -1) {
                    // It's a collection, but has no entries that match the OGNL expression for this element...
                    if(clonePoint == element) {
                        // If the clonePoint is the element itself, we remove it... we're done with it...
                        clonePoint.getParentNode().removeChild(clonePoint);
                    } else {
                        // If the clonePoint is not the element itself (it's a child element), leave it
                        // and check it again when we get to it...
                        resetClonePoint(clonePoint);
                    }
                } else if(collectionSize == 0) {
                    // It's a collection, but has no entries, remove it...
                    clonePoint.getParentNode().removeChild(clonePoint);
                } else if(collectionSize == 1) {
                    // It's a collection, but no need to clone coz we
                    // already have an entry for it...
                } else {
                    // It's a collection and we need to do some cloning
                    if(clonePoint != null) {
                        // We already have one, so decrement by one...
                        cloneCollectionTemplateElement(clonePoint, (collectionSize - 1), ognl);
                    } else {
                        logger.warn("Collection/array template element <" + element.getLocalName() + "> would appear to be invalid.  It doesn't contain any child elements.");
                    }
                }
            }
        }

        // Now do the same for the child elements...
        List<Node> children = YADOMUtil.copyNodeList(element.getChildNodes());
        for (Node node : children) {
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                expandMessage((Element) node, params);
            }
        }
    }

    private int calculateCollectionSize(String ognl, Map params) {
        // Try for an Object graph based collection...
        Object param = OGNLUtils.getParameter(ognl, params);
        if (param != null) {
            Class paramRuntime = param.getClass();
            if (paramRuntime.isArray()) {
                return ((Object[]) param).length;
            } else if (Collection.class.isAssignableFrom(paramRuntime)) {
                return ((Collection) param).size();
            }
        }

        // Try for <String, Object> collection based map entries...
        Set<Map.Entry> entries = params.entrySet();
        String collectionPrefix = ognl + "[";
        int maxIndex = -1;
        for (Map.Entry entry : entries) {
            Object keyObj = entry.getKey();
            if(keyObj instanceof String) {
                String key = (String)keyObj;
                if(key.startsWith(collectionPrefix)) {
                    int endIndex = key.indexOf(']', collectionPrefix.length());
                    String ognlIndexValString = key.substring(collectionPrefix.length(), endIndex);
                    try {
                        int ognlIndexVal = Integer.valueOf(ognlIndexValString);
                        maxIndex = Math.max(maxIndex, ognlIndexVal);
                    } catch(NumberFormatException e) {}
                }
            }
        }

        if(maxIndex != -1) {
            return maxIndex + 1;
        }

        // It's a collection, but nothing in this message for it collection...
        return -1;
    }

    private Element getClonePoint(Element element) {
        Comment comment;

        // Is it this element...
        comment = getCommentBefore(element);
        if(comment != null && comment.getTextContent().endsWith(SOAPUI_CLONE_COMMENT)) {
            comment.setTextContent(comment.getTextContent() + CLONED_POSTFIX);
            return element;
        }

        // Is it the first child element of this element...
        Element firstChildElement = getFirstChildElement(element);
        if(firstChildElement != null) {
            comment = getCommentBefore(firstChildElement);
            if(comment != null && comment.getTextContent().endsWith(SOAPUI_CLONE_COMMENT)) {
                comment.setTextContent(comment.getTextContent() + CLONED_POSTFIX);
                return firstChildElement;
            }
        }

        return null;
    }

    private void resetClonePoint(Element clonePoint) {
        Comment comment = getCommentBefore(clonePoint);

        if(comment == null) {
            throw new IllegalStateException("Call to reset a 'clonePoint' that doesn't have a comment before it.");
        }

        String commentText = comment.getTextContent();
        if(!commentText.endsWith(CLONED_POSTFIX)) {
            throw new IllegalStateException("Call to reset a 'clonePoint' that doesn't have a proper clone comment before it.");
        }

        comment.setTextContent(commentText.substring(0, commentText.length() - CLONED_POSTFIX.length()));
    }

    private Comment getCommentBefore(Element element) {
        Node sibling = element.getPreviousSibling();

        while(sibling != null) {
            if(sibling.getNodeType() == Node.COMMENT_NODE) {
                return (Comment) sibling;
            } else if(sibling.getNodeType() == Node.TEXT_NODE) {
                // continue...
                sibling = sibling.getPreviousSibling();
            } else {
                // It's an Element, CData, PI etc
                return null;
            }
        }

        return null;
    }

    private Element getFirstChildElement(Element element) {
        NodeList children = element.getChildNodes();
        int childCount = children.getLength();

        for(int i = 0; i < childCount; i++) {
            Node child = children.item(i);
            if(child.getNodeType() == Node.ELEMENT_NODE) {
                return (Element) child;
            }
        }

        return null;
    }

    /**
     * Clone a collection node.
     * <p/>
     * Note we have to frig with the OGNL expressions for collections/arrays because the
     * collection entry is represented by [0], [1] etc in the OGNL expression, not the actual
     * element name on the DOM e.g. collection node "order/items/item" (where "item" is the
     * actual collection entry) maps to the OGNL expression "order.items[0]" etc.
     *
     * @param element    The collection/array "entry" sub-branch.
     * @param cloneCount The number of times it needs to be cloned.
     * @param ognl       The OGNL expression for the collection/array. Not including the
     *                   indexing part.
     */
    private void cloneCollectionTemplateElement(Element element, int cloneCount, String ognl) {
        if (element == null) {
            return;
        }

        Element nextSibling = YADOMUtil.getNextSiblingElement(element);
        Node parent = element.getParentNode();

        element.setAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, OGNLUtils.JBOSSESB_SOAP_NS_PREFIX + OGNLUtils.OGNL_ATTRIB, ognl + "[0]");
        for (int i = 0; i < cloneCount; i++) {
            Element clone = (Element) element.cloneNode(true);

            clone.setAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, OGNLUtils.JBOSSESB_SOAP_NS_PREFIX + IS_CLONE_ATTRIB, "true");
            clone.setAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, OGNLUtils.JBOSSESB_SOAP_NS_PREFIX + OGNLUtils.OGNL_ATTRIB, ognl + "[" + Integer.toString(i + 1) + "]");
            if (nextSibling == null) {
                parent.appendChild(clone);
            } else {
                parent.insertBefore(clone, nextSibling);
            }
            nextSibling = YADOMUtil.getNextSiblingElement(clone);
        }
    }

    private void injectParameters(Element element, Map params, String soapNs) {
        NodeList children = element.getChildNodes();
        int childCount = children.getLength();

        for (int i = 0; i < childCount; i++) {
            Node node = children.item(i);

            if (childCount == 1 && node.getNodeType() == Node.TEXT_NODE) {
                if (node.getTextContent().equals("?")) {
                    String ognl = OGNLUtils.getOGNLExpression(element, soapNs);
                    Object param;

                    param = OGNLUtils.getParameter(ognl, params);

                    element.removeChild(node);
                    element.appendChild(element.getOwnerDocument().createTextNode(param.toString()));
                }
            } else if (node.getNodeType() == Node.ELEMENT_NODE) {
                injectParameters((Element) node, params, soapNs);
            }
        }

        element.removeAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, IS_CLONE_ATTRIB);
        element.removeAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, OGNLUtils.OGNL_ATTRIB);
    }

    private synchronized DocumentBuilder getDocBuilder() throws IOException {
        try {
            return docBuilderFactory.newDocumentBuilder();
        } catch (final ParserConfigurationException pce) {
            final IOException ioe = new IOException("Could not create document builder") ;
            ioe.initCause(pce) ;
            throw ioe ;
        }
    }
}
