/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.soap;

import org.jboss.mx.util.MBeanServerLocator;
import org.xml.sax.SAXException;

import javax.management.*;
import java.util.Map;
import java.util.Properties;
import java.io.IOException;

/**
 * Invoker the soapui-client Service.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SoapUIInvoker {

    private static final String[] buildRequestSig = new String[] {String.class.getName(), String.class.getName(), Map.class.getName(), Properties.class.getName(), String.class.getName(), String.class.getName()};
    private static final String[] getEndpointSig = new String[] {String.class.getName(), Properties.class.getName()};
    private MBeanServer mbeanServer;
    private ObjectName serviceName;

    public SoapUIInvoker() {
        // Lookup the MBean ObjectName for the soapUI service (in soapui-client.sar)...
        try {
            mbeanServer = MBeanServerLocator.locate();
            serviceName = new ObjectName("jboss.esb:service=SoapUIClientService");
        } catch (MalformedObjectNameException e) {
            throw new IllegalStateException("Unexpected exception.", e);
        }
    }

    /**
     * Build a SOAP request for the specified operation on the specified WSDL.
     *
     * @param wsdl            WSDL URL.
     * @param operation       Operation name.
     * @param params          Message parameter map.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @param smooksResource  {@link org.milyn.Smooks} transformation configuration resource file.
     *                        Null if no transformations are to be performed on the SOAP message before serializing it
     *                        for return.
     *  @param soapNs 		  the SOAP namespace. If null one of the defaults will be used:
     * 						  http://schemas.xmlsoap.org/soap/envelope/
     * 						  http://www.w3.org/2003/05/soap-envelope 
     * @return The SOAP Message.
     * @throws java.io.IOException                   Failed to load WSDL.
     * @throws UnsupportedOperationException Operation not supported on specified WSDL.
     * @throws org.xml.sax.SAXException                  Failed to parse the SOAP UI generated request message.
     */
    public String buildRequest(String wsdl, String operation, Map params, Properties httpClientProps, String smooksResource, String soapNs) throws IOException, UnsupportedOperationException, SAXException {
        try {
            return (String) mbeanServer.invoke(serviceName, "buildRequest", new Object[] {wsdl, operation, params, httpClientProps, smooksResource, soapNs}, buildRequestSig);
        } catch (InstanceNotFoundException e) {
            throw new UnsupportedOperationException("SOAP UI Client Service not found under name '" + serviceName.getCanonicalName() + "'.  This service must be deployed before this action can be used.", e);
        } catch (MBeanException e) {
            rethrowException(e);
        } catch (ReflectionException e) {
            rethrowException(e);
        }

        throw new RuntimeException("Unexpected code exit point reached!");
    }

    /**
     * Get the 1st endpoint from the specified WSDL.
     *
     * @param wsdl WSDL URL.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @return The operation endpoint URL.
     * @throws IOException Failed to load WSDL.
     */
    public String getEndpoint(String wsdl, Properties httpClientProps) throws IOException {
        try {
            return (String) mbeanServer.invoke(serviceName, "getEndpoint", new Object[] {wsdl, httpClientProps}, getEndpointSig);
        } catch (InstanceNotFoundException e) {
            throw new UnsupportedOperationException("SOAP UI Client Service not found under name '" + serviceName.getCanonicalName() + "'.  This service must be deployed before this action can be used.", e);
        } catch (MBeanException e) {
            if(e.getCause() instanceof IOException) {
                throw (IOException)e.getCause();
            }
            throw new RuntimeException(e);
        } catch (ReflectionException e) {
            if(e.getCause() instanceof IOException) {
                throw (IOException)e.getCause();
            }
            throw new RuntimeException(e);
        }
    }

    private void rethrowException(Throwable e) throws IOException, UnsupportedOperationException, SAXException {
        Throwable cause = e.getCause();

        if(cause instanceof IOException) {
            throw (IOException)cause;
        } else if(cause instanceof UnsupportedOperationException) {
            throw (UnsupportedOperationException)cause;
        } else if(cause instanceof SAXException) {
            throw (SAXException)cause;
        } else {
            throw new RuntimeException(e);
        }
    }

}
