/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.routing.cbr;

import java.io.ByteArrayInputStream;
import java.io.StringReader;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.xml.sax.InputSource;

/**
 * Domain Specific Language helper. Right now this supports the use of XPath, but this class can
 * be beefed up upo to use other technologies as well.
 * 
 * @author kstam@redhat.com
 *
 */
public class DslHelper 
{
	private static Logger log = Logger.getLogger(DslHelper.class);
	/** XPath instance */
	private static XPathFactory xpf = XPathFactory.newInstance();
    private static MessagePayloadProxy payloadProxy;

    static {
        payloadProxy = new MessagePayloadProxy(new ConfigTree("config"),
                      new String[] {BytesBody.BYTES_LOCATION},
                      new String[] {BytesBody.BYTES_LOCATION});
    }

    /**
	 * Uses XPath to look for the occurence of a certain tag, specific in the xpath expression.
	 * 
	 * @param message - the ESB Message which body content will be used.
	 * @param xpathExp - XPath expression to find a node.
	 * @return true if the node is found and false in all other cases.
	 * @throws XPathExpressionException
	 */
	public static boolean xmlContentMatches(Message message, String xpathExp) throws XPathExpressionException 
	{
		if (log.isDebugEnabled()) {
			log.debug("Trying to match xpath: '" + xpathExp +  "' in message=" + message);
		}
		XPath xpath = xpf.newXPath();
        InputSource inputSource = getInputSource(message);
        Object node = xpath.evaluate(xpathExp, inputSource, XPathConstants.NODE);
		if (log.isDebugEnabled()) {
			log.debug("Found node=" + node);
		}
		return node != null;
	}

    /**
	 * Uses XPath to look for the occurence of a certain tag, specific in the xpath expression.
	 *
	 * @param message - the ESB Message which body content will be used.
	 * @param xpathExp - XPath expression to find a node.
	 * @param value - used to compare against the result found using the XPath expression.
	 * @return true if equal, false in all other cases.
	 * @throws XPathExpressionException
	 */
	public static boolean xmlContentEquals(Message message, String xpathExp, String value) throws XPathExpressionException
	{
		if (log.isDebugEnabled()) {
			log.debug("Trying to match xpath: '" + xpathExp +  "' in message=" + message.getBody().get());
		}
		XPath xpath = xpf.newXPath();
        InputSource inputSource = getInputSource(message);
		String nodeValue = (String) xpath.evaluate(xpathExp, inputSource, XPathConstants.STRING);
		if (log.isDebugEnabled()) {
			log.debug("Found nodeValue=" + nodeValue + " which is matched to given value=" + value);
		}
		return value.equals(nodeValue);
	}

    /**
	 * Uses XPath to look for the occurence of a certain tag, specific in the xpath expression.
	 *
	 * @param message - the ESB Message which body content will be used.
	 * @param xpathExp - XPath expression to find a node.
	 * @param value - used to compare against the result found using the XPath expression.
	 * @return true if node (returned by the xpath expression) is greater than the current value, false in all other cases.
	 * @throws XPathExpressionException
	 */
	public static boolean xmlContentGreaterThan(Message message, String xpathExp, String value) throws XPathExpressionException
	{
		String nodeValue=null;
		double doubleValue=0;
		double doubleNodeValue=0;

		if (log.isDebugEnabled()) {
			log.debug("Trying to match xpath: '" + xpathExp +  "' in message=" + message.getBody().get());
		}
		XPath xpath = xpf.newXPath();
		InputSource inputSource = getInputSource(message);	
		nodeValue = (String) xpath.evaluate(xpathExp, inputSource, XPathConstants.STRING);if (log.isDebugEnabled()) {
			log.debug("Found nodeValue=" + nodeValue + " which is matched to given value=" + value);
		}
		if (nodeValue!=null && !"".equals(nodeValue)) {

			try {
				doubleValue = Double.parseDouble(value);
			} catch (NumberFormatException ne) {
				log.equals("Could not parse value=" + doubleValue + " to double");
			}
			try {
				doubleNodeValue = Double.parseDouble(nodeValue);
			} catch (NumberFormatException ne) {
				log.equals("Could not parse nodeValue=" + doubleNodeValue + " to double");
			}
			if (doubleNodeValue > doubleValue) {
				return true;
			}
		}
		return false;
	}

    /**
	 * Uses XPath to look for the occurence of a certain tag, specific in the xpath expression.
	 *
	 * @param message - the ESB Message which body content will be used.
	 * @param xpathExp - XPath expression to find a node.
	 * @param value - used to compare against the result found using the XPath expression.
	 * @return true if node (returned by the xpath expression) is less than the current value, false in all other cases.
	 * @throws XPathExpressionException
	 */
	public static boolean xmlContentLessThan(Message message, String xpathExp, String value) throws XPathExpressionException
	{
		String nodeValue=null;
		double doubleValue=0;
		double doubleNodeValue=0;

		if (log.isDebugEnabled()) {
			log.debug("Trying to match xpath: '" + xpathExp +  "' in message=" + message.getBody().get());
		}
		XPath xpath = xpf.newXPath();
		InputSource inputSource = getInputSource(message);
		nodeValue = (String) xpath.evaluate(xpathExp, inputSource, XPathConstants.STRING);if (log.isDebugEnabled()) {
			log.debug("Found nodeValue=" + nodeValue + " which is matched to given value=" + value);
		}
		if (nodeValue!=null && !"".equals(nodeValue)) {

			try {
				doubleValue = Double.parseDouble(value);
			} catch (NumberFormatException ne) {
				log.equals("Could not parse value=" + doubleValue + " to double");
			}
			try {
				doubleNodeValue = Double.parseDouble(nodeValue);
			} catch (NumberFormatException ne) {
				log.equals("Could not parse nodeValue=" + doubleNodeValue + " to double");
			}
			if (doubleNodeValue < doubleValue) {
				return true;
			}
		}
		return false;
	}

    private static InputSource getInputSource(Message message) throws XPathExpressionException {
        Object payload;

        try {
            payload = payloadProxy.getPayload(message);
        } catch (MessageDeliverException e) {
            throw new XPathExpressionException(e);
        }

        if(payload instanceof byte[]) {
            return new InputSource(new ByteArrayInputStream((byte[]) payload));
        } else if(payload instanceof String) {
            return new InputSource(new StringReader((String) payload));
        } else {
            throw new XPathExpressionException("Unsupport expression input object type: " + payload.getClass().getName());
        }
    }
}
	
