/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 */

package org.jboss.soa.esb.testutils;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.couriers.JmsCourier;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierTimeoutException;

import javax.jms.Message;
import javax.jms.ObjectMessage;
import javax.jms.JMSException;
import javax.naming.Context;
import java.io.Serializable;
import java.net.URISyntaxException;

/**
 * JMS Client utilities.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JMSUtil {

	private static Logger logger = Logger.getLogger(JMSUtil.class);

    /**
     * Receive a message on the specified JMS Queue.
     * @param queue The JMS Queue name.
     */
    public static javax.jms.Message receiveMessageOnQueue(String queue, long timeout) {
        AssertArgument.isNotNullAndNotEmpty(queue, "queue");

        JMSEpr epr = new JMSEpr(JMSEpr.QUEUE_TYPE, queue, "ConnectionFactory");
        return receiveMessage(epr, timeout);
    }

    /**
     * Receive a message on the specified JMS Topic.
     * @param topic The JMS Topic name.
     */
    public static javax.jms.Message receiveMessageOnTopic(String topic, long timeout) {
        AssertArgument.isNotNullAndNotEmpty(topic, "topic");

        JMSEpr epr = new JMSEpr(JMSEpr.TOPIC_TYPE, topic, "ConnectionFactory");
        return receiveMessage(epr, timeout);
    }

    /**
     * Send the supplied message to the specified JMS Queue.
     * @param message The message to send.
     * @param queue The JMS Queue name.
     * @param properties A property set for the JMS Message.  Can be null.
     */
    public static void sendMessageToQueue(Serializable message, String queue, KeyValuePair[] properties) {
        AssertArgument.isNotNull(message, "message");
        AssertArgument.isNotNullAndNotEmpty(queue, "queue");

        JMSEpr epr = new JMSEpr(JMSEpr.QUEUE_TYPE, queue, "ConnectionFactory");
        deliverMessage(message, epr, properties);
    }

    /**
     * Send the supplied message to the specified JMS Topic.
     * @param message The message to send.
     * @param topic The JMS Topic name.
     * @param properties A property set for the JMS Message.  Can be null.
     */
    public static void sendMessageToTopic(Serializable message, String topic, KeyValuePair[] properties) {
        AssertArgument.isNotNull(message, "message");
        AssertArgument.isNotNullAndNotEmpty(topic, "topic");

        JMSEpr epr = new JMSEpr(JMSEpr.TOPIC_TYPE, topic, "ConnectionFactory");
        deliverMessage(message, epr, properties);
    }

    private static Message receiveMessage(JMSEpr epr, long timeout) {
        JmsCourier courier = null;

        addJndiExtensions(epr);

        try {
            courier = new JmsCourier(epr, true);
        } catch (CourierException e) {
            throw new RuntimeException("Failed to create pickup courier for epr '" + epr + "': " + e.getMessage());
        }

        try {
            return courier.pickupPayload(timeout);
        } catch (CourierException e) {
            throw new RuntimeException("Failed to pickup message on JMS epr '" + epr + "': " + e.getMessage());
        } catch (CourierTimeoutException e) {
            throw new RuntimeException("Failed to pickup message on JMS epr '" + epr + "': " + e.getMessage());
        }
    }

    private static void deliverMessage(Serializable message, JMSEpr epr, KeyValuePair[] properties) {
        String destType = null;
        String destName = null;
        JmsCourier courier = null;
        ObjectMessage jmsMessage = null;

        addJndiExtensions(epr);

        try {
            destType = epr.getDestinationType();
            destName = epr.getDestinationName();
        } catch (URISyntaxException e) {
            // An exception on a getter method??? That's an interesting/new approach!!
            throw new RuntimeException("Totally unexpected exception from a getter method: " + e.getMessage());
        }

        try {
            courier = new JmsCourier(epr);
        } catch (CourierException e) {
            throw new RuntimeException("Failed to create JMSCourier to JMS " + destType + " '" + destName + "': " + e.getMessage());
        }

        try {
            try {
                jmsMessage = courier.getJmsSession(epr.getAcknowledgeMode()).createObjectMessage(message);
            } catch (CourierException e) {
                throw new RuntimeException("Failed to get JMS Session for sending to JMS " + destType + " '" + destName + "': " + e.getMessage());
            } catch (JMSException e) {
                throw new RuntimeException("Failed to create JMS Message for sending to JMS " + destType + " '" + destName + "': " + e.getMessage());
            }

            if(properties != null) {
                try {
                    setStringProperties(jmsMessage, properties);
                } catch (JMSException e) {
                    throw new RuntimeException("Failed to set properties on the JMS Message to JMS " + destType + " '" + destName + "': " + e.getMessage());
                }
            }

            try {
                courier.deliver(jmsMessage);
            } catch (CourierException e) {
                throw new RuntimeException("Failed to deliver JMS Message to JMS " + destType + " '" + destName + "': " + e.getMessage());
            }
        } finally {
            courier.cleanup();
        }
    }

    private static void addJndiExtensions(JMSEpr epr) {
        epr.getAddr().addExtension(Context.INITIAL_CONTEXT_FACTORY, "org.jnp.interfaces.NamingContextFactory");
        epr.getAddr().addExtension(Context.PROVIDER_URL, "localhost");
    }

    private static void setStringProperties(Message msg, KeyValuePair[] properties) throws JMSException {
		if(properties == null) {
			return;
		}

		for(KeyValuePair property : properties) {
			msg.setStringProperty(property.getKey(), property.getValue());
		}
	}

    public static void main(String[] args) {
        if(args.length < 3) {
            throw new RuntimeException("Invalid arg list. 3 args expected. arg1: destName, arg2: destType (queue/topic), arg3: message");
        }

        String destName = args[0];
        String destType = args[1];
        String message = args[2];

        if(destType.equals("queue")) {
            sendMessageToQueue(message, destName, null);
        } else if(destType.equals("topic")) {
            sendMessageToTopic(message, destName, null);
        } else {
            throw new RuntimeException("Invalid 2nd arg value for destType. Must be one of 'queue' or 'topic'.");
        }
    }
}
