/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.schedule;

import junit.framework.TestCase;
import org.jboss.soa.esb.testutils.ESBConfigUtil;
import org.jboss.soa.esb.parameters.ParamRepositoryException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.mock.MockAction;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.xml.sax.SAXException;
import org.quartz.impl.StdSchedulerFactory;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;

import java.io.UnsupportedEncodingException;
import java.io.InputStream;
import java.io.IOException;
import java.util.Properties;

/**
 * Scheduling unit tests.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SchedulingUnitTest extends TestCase {


    protected void setUp() throws Exception {
        MockScheduledEventMessageComposer.reset();
        MockScheduledEventListener.reset();
        MockAction.message = null;
    }

    public void test_listener_config() throws UnsupportedEncodingException {
        ESBConfigUtil configUtil = new ESBConfigUtil(getClass().getResourceAsStream("config-01.xml"));
        String expected = StreamUtils.readStreamString(getClass().getResourceAsStream("expected-config-01-listener.xml"), "UTF-8");

        assertEquals(expected, configUtil.getListenerConfig("simple-schedule-listener").toString());
    }

    public void test_simple_schedule_01() throws UnsupportedEncodingException, ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        runTestConfig("config-01.xml", 5000);

        // Check that the scheduling happened as defined in the config...
        assertNotNull(MockScheduledEventMessageComposer.config);
        assertEquals(4, MockScheduledEventMessageComposer.composedMessages.size());
        assertTrue(MockScheduledEventMessageComposer.uninitialised);
        assertEquals(4, MockScheduledEventMessageComposer.completedMessage.size());

        // Check that the action pipeline was executed as expeected...
        assertNotNull(MockAction.message);
    }

    public void test_simple_schedule_02() throws UnsupportedEncodingException, ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        runTestConfig("config-02.xml", 5000);

        // Just check that the schedule was triggered more than once...
        assertTrue(MockScheduledEventMessageComposer.composedMessages.size() > 1);
    }

    public void test_simple_schedule_03() throws UnsupportedEncodingException, ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        runTestConfig("config-03.xml", 5000);

        // Just check that the schedule was triggered more than once...
        assertTrue(MockScheduledEventMessageComposer.composedMessages.size() > 1);
    }

    public void test_simple_schedule_03_1() throws UnsupportedEncodingException, ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        MockScheduledEventListener.sleep = 6000;
        runTestConfig("config-03.1.xml", 7000);

        // Just check that the schedule wasn't run concrurrently...
        assertTrue(MockScheduledEventListener.execCount < 3);
    }

    public void test_simple_schedule_04() throws UnsupportedEncodingException, ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        runTestConfig("config-04.xml", 5000);

        // Just check that the schedule was triggered more than once...
        assertTrue(MockScheduledEventMessageComposer.composedMessages.size() > 1);
    }

    public void test_simple_schedule_04_JBESB_843() throws UnsupportedEncodingException, ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        runTestConfig("config-04-JBESB-843.xml", 5000);

        // Just check that the schedule was triggered more than once...
        assertTrue(MockScheduledEventMessageComposer.composedMessages.size() > 1);
    }

    public void test_simple_schedule_05() throws UnsupportedEncodingException, ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        runTestConfig("config-05.xml", 5000);

        // Just check that the schedule was triggered more than once...
        assertTrue(MockScheduledEventMessageComposer.composedMessages.size() > 1);
    }

    public void test_simple_schedule_06() throws UnsupportedEncodingException, ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        runTestConfig("config-06.xml", 5000);

        // Check that the scheduling happened as defined in the config...
        assertNotNull(MockScheduledEventListener.config);
        assertTrue(MockScheduledEventListener.onSchedule);
        assertTrue(MockScheduledEventListener.uninitialised);
    }

    private void runTestConfig(String configName, long upTime) throws ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        ESBConfigUtil configUtil = new ESBConfigUtil(getClass().getResourceAsStream(configName));

        configUtil.startController();
        Thread.sleep(upTime);
        configUtil.stopController();
    }
}
