/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.gateway;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.common.tests.BaseTest;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.body.content.BytesBody;

public class FileGatewayListenerUnitTest extends BaseTest
{
	private Logger log = Logger.getLogger( FileGatewayListenerUnitTest.class );

    private File tmpDir = new File(System.getProperty("user.dir")) ;
    private File dir1 = new File(tmpDir, FileGatewayListenerUnitTest.class.getSimpleName());
    private File dir2 = new File(tmpDir, FileGatewayListenerUnitTest.class.getSimpleName());
    private File file1 = new File(dir1, "file1");
    private final File file2 = new File(dir2, "file2");

    public FileGatewayListenerUnitTest ()
	{
    }
	
	public void setUp()
	{
		MockRegistry.install();
        cleanupTemps();
        dir1.mkdirs();
        dir2.mkdirs();
    }

    public void tearDown()
	{
        cleanupTemps();
		MockRegistry.uninstall();
	}

    private void cleanupTemps() {
        file1.delete();
        file2.delete();
        dir1.delete();
        dir2.delete();
    }

    public void testGateway () throws Exception
	{
        ConfigTree tree = createTestListenerConfig();
        FileGatewayListener gateway = new FileGatewayListener(tree);

        boolean exception = false;
		
		try
		{
			gateway.seeIfOkToWorkOnDir(new File("foobarDir"));
		}
		catch (ConfigurationException ex)
		{
			exception = true;
		}
		
		if (!exception)
			fail();
		
		try
		{
			gateway.doInitialise();
		}
		catch (ManagedLifecycleException ex)
		{
		}
		catch (Exception ex)
		{
			log.error(ex);
			
			fail();
		}
		
		exception = false;
		
		try
		{
			gateway.stop();
		}
		catch (ManagedLifecycleException ex)
		{
			exception = true;
		}
		catch (Exception ex)
		{
			fail();
		}
		
		gateway.onSchedule();
		
		LocalFileMessageComposer<File> messageComposer = new LocalFileMessageComposer<File>();

        try {
            messageComposer.setConfiguration(new ConfigTree("config"));
            messageComposer.compose(new File("hello"));
            fail("Expected exception on non-existant file.");
        } catch(MessageDeliverException e) {
            // expected
        }

		File testFile = null;
		File anTestFile = null;
		
		try
		{
			testFile = File.createTempFile("foo", "testFile");
			
			String testString = "Hello World";
			FileOutputStream stream = new FileOutputStream(testFile);
			
			stream.write(testString.getBytes());
			stream.close();

			Message message = messageComposer.compose(testFile);
			
			File nFile = null;
			
			try
			{
				byte[] content = (byte[]) message.getBody().get();
				String value = new String(content);
				
				if (!value.equals(testString))
					fail();
				
				if (!gateway.deleteFile(testFile))
					fail();
				
				testFile = File.createTempFile("foo", "testFile",gateway._inputDirectory);
				anTestFile = File.createTempFile("bar", "testFile",gateway._inputDirectory);				

				tree.setAttribute("inputSuffix", "testFile");
				gateway = new FileGatewayListener(tree);
                File[] files = gateway.getFileList();
				
				if (files.length != 2)
				{
					for (int i = 0; i < files.length; i++)
						System.err.println(files[i]);
				
					fail();	
				}
				
				exception = false;
				
				nFile = new File("xyzzy");
				
				if (gateway.renameFile(anTestFile, nFile))
				{
					anTestFile = nFile;
				}
				else
					fail();
			}
			catch (Exception ex)
			{
				fail();
			}
			finally
			{
				nFile.delete();
			}
		}
		catch (Throwable ex)
		{
			ex.printStackTrace();
			fail(ex.getMessage());
		}
		finally
		{
			testFile.delete();
			anTestFile.delete();
		}
	}

    public void test_rename_ok() throws GatewayException, IOException, RegistryException, ConfigurationException {
        ConfigTree tree = createTestListenerConfig();

        FileGatewayListener gateway = new FileGatewayListener(tree) {
            protected File getWorkFileName(File fileIn, String suffix) {
                return file2;
            }
        };

        file1.createNewFile();
        assertTrue(file1.exists());
        assertTrue(!file2.exists());

        File workingFile = gateway.setFileWorking(file1);
        assertNotNull(workingFile);
        assertTrue(!file1.exists());
        assertTrue(file2.exists());
        assertTrue(file2.equals(workingFile));
    }

    public void test_rename_no_from_file() throws GatewayException, IOException, RegistryException, ConfigurationException {
        ConfigTree tree = createTestListenerConfig();

        FileGatewayListener gateway = new FileGatewayListener(tree) {
            protected File getWorkFileName(File fileIn, String suffix) {
                return file2;
            }
        };

        assertTrue(!file1.exists());
        assertTrue(!file2.exists());

        assertNull(gateway.setFileWorking(file1));
    }

    public void test_rename_to_file_exists() throws GatewayException, IOException, RegistryException, ConfigurationException {
        ConfigTree tree = createTestListenerConfig();

        FileGatewayListener gateway = new FileGatewayListener(tree) {
            protected File getWorkFileName(File fileIn, String suffix) {
                return file2;
            }
        };

        file2.createNewFile();
        assertTrue(!file1.exists());
        assertTrue(file2.exists());

        assertNull(gateway.setFileWorking(file1));
    }

    private ConfigTree createTestListenerConfig() throws MalformedURLException, ConfigurationException, RegistryException, GatewayException {
        ConfigTree tree = new ConfigTree("test");
        final File tmpDir = new File(System.getProperty("user.dir")) ;
        final String tmpDirForm = tmpDir.toURL().toExternalForm() ;

        tree.setAttribute("inputDir", tmpDirForm);
        tree.setAttribute("target-service-category", "Example");
        tree.setAttribute("target-service-name", "Test");
        tree.setAttribute("gatewayClass", "org.jboss.soa.esb.listeners.gateway.FileGatewayListener");
        tree.setAttribute("inputSuffix", "dummy");
        tree.setAttribute("workSuffix", "work");
        tree.setAttribute("postDelete", "true");
        tree.setAttribute(ListenerTagNames.POLL_LATENCY_SECS_TAG, "abcd");

        return tree;
    }
}
