/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import javax.jms.JMSException;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.notification.jms.JMSPropertiesSetter;
import org.junit.Before;
import org.junit.Test;
import org.mockejb.jms.MockQueue;
import org.mockejb.jms.TextMessageImpl;

/**
 * Unit test for DefaultESBPropertiesSetter
 * </p>
 * 
 * @author <a href="mailto:daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 * @since 4.2
 */
public class DefaultESBPropertiesSetterUnitTest
{
	@SuppressWarnings ("unused")
	private Logger log = Logger .getLogger( DefaultESBPropertiesSetterUnitTest.class );
	
	private final static String jmsMessageID = "123456780";
	private final static String jmsCorrelationID = "YYXX-123456780-GG";
	private ESBPropertiesSetter strategy;
	private javax.jms.Message fromJMSTextMessage;
	private Message toESBMessage;
	
	@Test (expected = JMSException.class )
	public void setPropertiesFromJMSMessage_MessageID_negative() throws JMSException
	{
		log.debug( "The following Exception is expected: ");
		fromJMSTextMessage.setJMSMessageID( "  -- // == " );
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
	}
	
	@Test
	public void setPropertiesFromJMSMessage_MessageID() throws JMSException
	{
		fromJMSTextMessage.setJMSMessageID( jmsMessageID );
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
		
		URI messageID = toESBMessage.getHeader().getCall().getMessageID();
		assertEquals( jmsMessageID,  messageID.toString() );
	}
	
	@Test
	public void setPropertiesFromJMSMessage_CorrelationID() throws JMSException
	{
		fromJMSTextMessage.setJMSCorrelationID( jmsCorrelationID );
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
		
		assertEquals( jmsCorrelationID, toESBMessage.getHeader().getCall().getRelatesTo().getFragment() );
	}
	
	@Test
	public void setPropertiesFromJMSMessage_ReplyTo() throws JMSException
	{
		final String destinationName = "testDest";
		final MockQueue queue = new MockQueue( destinationName );
		fromJMSTextMessage.setJMSReplyTo( queue );
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
		
		final EPR replyToEPR = toESBMessage.getHeader().getCall().getReplyTo();
		
		assertEquals( "jms://localhost/testDest", replyToEPR.getAddr().getAddress() );
		
		final String connectionFactory = replyToEPR.getAddr().getExtensionValue( JMSEpr.CONNECTION_FACTORY_TAG );
		assertEquals ( "ConnectionFactory", connectionFactory );
	}
	
	@Test
	public void setPropertiesFromJMSMessage_Expiration() throws JMSException, InterruptedException
	{
		final long ttl = 5000l;
		final long expirationTime = System.currentTimeMillis() + ttl;
		fromJMSTextMessage.setJMSExpiration( expirationTime );
		
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
		
		Long actualExpiration = (Long)toESBMessage.getProperties().getProperty( JMSPropertiesSetter.JMS_EXPIRATION );
		assertTrue( actualExpiration != 0 );
		assertTrue( actualExpiration > 0 );
	}
	
	@Test
	public void setPropertiesFromJMSMessage_withProperties() throws JMSException
	{
		final String propertyKey = "myProperty";
		final String propertyValue = "myPropertyValue";
		fromJMSTextMessage.setStringProperty( propertyKey, propertyValue );
		
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
		
		assertEquals( propertyValue, toESBMessage.getProperties().getProperty( propertyKey )) ;
	}
	
	@Test
	public void setPropertiesFromJMSMessage_withJMSRedelivererdProperty_false() throws JMSException
	{
		boolean redelivered = false;
		fromJMSTextMessage.setJMSRedelivered( redelivered );
		
		strategy.setPropertiesFromJMSMessage( fromJMSTextMessage , toESBMessage );
		
		assertEquals( redelivered, toESBMessage.getProperties().getProperty( JMSPropertiesSetter.JMS_REDELIVERED )) ;
	}
	
	@Before
	public void setup()
	{
		fromJMSTextMessage = new TextMessageImpl();
		toESBMessage = MessageFactory.getInstance().getMessage();
		strategy = new DefaultESBPropertiesSetter();
	}
	
	/*
	 * Just here to help Ant to find annotated test.
	 */
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( DefaultESBPropertiesSetterUnitTest.class );
	}	

}
