/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners;

import java.io.FileInputStream;
import java.net.URI;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.UUID;

import org.jboss.internal.soa.esb.couriers.DeliverOnlyCourier;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleController;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;

public class ListenerManagerJDBCUnitTest extends ListenerManagerFileUnitTest
{
	public ListenerManagerJDBCUnitTest()
	{
		_file = "listenerJdbc.xml";
	}

	public void setUp()
	{
		_logger.info("Writing temp files to " + TMP_DIR);

		clearMessages() ;
		
		try
		{
			Statement stmt = getDbConnection().createStatement();

			try
			{
				stmt.executeUpdate("DROP TABLE esb_messages");
			}
			catch (Exception e)
			{
				// Ignore
			}

			stmt.executeUpdate("CREATE TABLE esb_messages (message_id varchar NOT NULL, message varchar, status varchar, insert_timestamp bigint, CONSTRAINT pkey_esb_messages PRIMARY KEY (message_id))");
		}
	catch (SQLException ex)
		{
			_logger.error(ex);

			fail();
		}

		// initialize registry
		runBeforeAllTests();
	}
	
	protected void oneTest () throws Exception
	{
		// Write wome messages to EPR obtained from configuration file
		String configFile = getClass().getResource(_file).getFile();
		ConfigTree tree = ConfigTree.fromInputStream(new FileInputStream(
				configFile));
		ConfigTree eprElement = tree.getAllChildren()[0].getFirstChild("EPR");
		
		eprElement.setAttribute(ListenerTagNames.URL_TAG, getDbUrl());
		eprElement.setAttribute("driver", getDbDriver());
		eprElement.setAttribute("username", getDbUser());
		eprElement.setAttribute("password", getDbPassword());
		
		EPR toEPR = ListenerUtil.assembleEpr(eprElement);

		if (toEPR instanceof JDBCEpr)
		{
			// OK, so ignore
		}
		else
			fail();
		
		String THE_TEXT = "___Config=" + _file + "___ Message Content ___";

		int howMany = 10; // how many messages do you want to send before the
							// listener comes up
		
		DeliverOnlyCourier sender = CourierFactory.getCourier(toEPR);
		
		Message message = MessageFactory.getInstance().getMessage();
		message.getHeader().setCall(new Call(toEPR));
		message.getBody().add(THE_TEXT.getBytes());
		
		for (int i1 = 0; i1 < howMany; i1++)
		{
			URI uri = new URI(UUID.randomUUID().toString());
			message.getHeader().getCall().setMessageID(uri);
			sender.deliver(message);
		}

		//		 launch listener manager in a child thread
		final ConfigTree newTree = ConfigTree.fromInputStream(new FileInputStream(
				configFile));
		
		eprElement = newTree.getAllChildren()[0].getFirstChild("EPR");
		
		eprElement.setAttribute(ListenerTagNames.URL_TAG, getDbUrl());
		eprElement.setAttribute("driver", getDbDriver());
		eprElement.setAttribute("username", getDbUser());
		eprElement.setAttribute("password", getDbPassword());

                final List<ManagedLifecycle> instances = LifecycleUtil.getListeners(newTree) ;
                final ManagedLifecycleController controller = new ManagedLifecycleController(instances) ;
                controller.start() ;
                
		_logger.debug(" All child listeners ready");

		// JUST FOR THIS TEST:
		// Give your listener some time to process queued messages (see howMany
		// above)
		// Time allowed, and maxThreads in config file will impact how many
		// messages
		// will be processed, and how many will remain unprocessed
		
		for (int count = 0 ; count < howMany ; count++)
		{
			final String response = getMessage(10000) ;
			assertNotNull("getMessage timeout", response) ;
			assertEquals(THE_TEXT, response);
		}

		_logger.debug("going to stop");
		controller.stop();
		_logger.debug("back from stop");
	}

}