/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.scripting;

import junit.framework.TestCase;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;

/**
 * Unit tests for GroovyActionProcessor.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class GroovyActionProcessorUnitTest extends TestCase {

    public void test_script_file() throws ConfigurationException, ActionLifecycleException, ActionProcessingException {
        ConfigTree config = getConfig("test.groovy");
        String messageContents = "Hello World!";

        config.setAttribute("messageContents", messageContents);

        GroovyActionProcessor processor = new GroovyActionProcessor(config);
        Message message = MessageFactory.getInstance().getMessage();

        assertProcessingOK(processor, message, messageContents, null);
    }

    public void test_script_caching_on() throws ConfigurationException, ActionLifecycleException, ActionProcessingException {
        ConfigTree config = getConfig("test.groovy");
        GroovyActionProcessor processor = new GroovyActionProcessor(config);
        Message message = MessageFactory.getInstance().getMessage();

        processor.initialise();
        // Test script caching...
        assertTrue("script should not be null", processor.getScript(message) != null);
        assertTrue("script should be cached", processor.getScript(message) == processor.getScript(message));                
    }

    public void test_script_caching_off() throws ConfigurationException, ActionLifecycleException, ActionProcessingException {
        ConfigTree config = getConfig("test.groovy");
        GroovyActionProcessor processor;
        Message message = MessageFactory.getInstance().getMessage();

        config.setAttribute("cacheScript", "false");
        processor = new GroovyActionProcessor(config);

        processor.initialise();
        // Test script caching...
        assertTrue("script should not be null", processor.getScript(message) != null);
        assertTrue("script should not be cached", processor.getScript(message) != processor.getScript(message));
    }

    public void test_script_inlined_fail() throws ConfigurationException, ActionLifecycleException, ActionProcessingException {
        ConfigTree config = new ConfigTree("<config/>");
        String messageContents = "Hello World - Inlined!";

        config.setAttribute("messageContents", messageContents);

        GroovyActionProcessor processor = new GroovyActionProcessor(config);
        Message message = MessageFactory.getInstance().getMessage();

        // Set the script as the message task object...
        message.getBody().add("import org.jboss.soa.esb.message.*\n" +
                "message.getBody().add(config.getAttribute(\"messageContents\").getBytes());");

        assertProcessingOK(processor, message, messageContents, "'script' not configured on the action and message based scripting is not enabled ('supportMessageBasedScripting=false').");
    }

    public void test_script_inlined_success() throws ConfigurationException, ActionLifecycleException, ActionProcessingException {
        ConfigTree config = new ConfigTree("<config/>");
        String messageContents = "Hello World - Inlined!";

        config.setAttribute("messageContents", messageContents);
        config.setAttribute("supportMessageBasedScripting", "true");

        GroovyActionProcessor processor = new GroovyActionProcessor(config);
        Message message = MessageFactory.getInstance().getMessage();

        // Set the script as the message task object...
        message.getBody().add("import org.jboss.soa.esb.message.*\n" +
                "message.getBody().add(config.getAttribute(\"messageContents\").getBytes());");

        assertProcessingOK(processor, message, messageContents, null);
    }

    private void assertProcessingOK(GroovyActionProcessor processor, Message message, String messageContents, String errorMessage) throws ActionLifecycleException, ActionProcessingException {
        try {
            processor.initialise();
            processor.process(message);
            byte[] bodyContents = (byte[]) message.getBody().get();
            assertNotNull("Expected body contents to be set.", bodyContents);
            assertEquals(messageContents, new String(bodyContents));
            if(errorMessage != null) {
                fail("Expected error message: \"" + errorMessage + "\".");
            }
        } catch(ActionLifecycleException e) {
            assertEquals(errorMessage, e.getMessage());
        }
    }

    private ConfigTree getConfig(String script) {
        ConfigTree config = new ConfigTree("<config/>");

        config.setAttribute("script", "/org/jboss/soa/esb/actions/scripting/" + script);
        
        return config;
    }
}
