/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.routing;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Queue;
import javax.jms.TextMessage;
import javax.jms.Topic;
import javax.naming.NamingException;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.mockejb.jms.ObjectMessageImpl;
import org.mockejb.jms.TextMessageImpl;

/**
 * 
 * @author John Doe
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 *
 */
public class JmsRouterIntegrationTest 
{
	@SuppressWarnings ( "unused")
	private Logger log = Logger.getLogger( JmsRouterIntegrationTest.class );
	
	private final String messageID = "1234-junittest";
	private final String bodyContent = "hello";
	private Message msg;
	private ConfigTree tree;
	
	@Before
	public void setup() throws URISyntaxException
	{
		msg = createESBMessageObject( messageID, bodyContent );
		tree = createConfigTree();
	}
	
	@Test
	public void process_unwrap_false() throws ConfigurationException, NamingException, JMSException, ActionProcessingException, URISyntaxException
	{
		MockJMSRouter router = new MockJMSRouter(tree);
		router.route( msg );
		
		assertProcessContract( messageID, msg, router );
	}
	
	@Test
	public void process_unwrap_true() throws ConfigurationException, NamingException, JMSException, ActionProcessingException, URISyntaxException
	{
		tree.setAttribute( "unwrap", "true" );
		MockJMSRouter router = new MockJMSRouter( tree );
		
		router.route( msg );
		
		assertProcessContract( messageID, msg, router );
		final javax.jms.Message jmsMessage = ((MockJMSRouter) router).getJmsMessage();
		
		assertTrue ( jmsMessage instanceof TextMessage );
		final javax.jms.TextMessage textMessage = (TextMessage) jmsMessage;
		
		assertEquals ( textMessage.getJMSCorrelationID(), messageID );
		assertEquals ( textMessage.getText(), bodyContent );
	}

	@Test ( expected = ActionProcessingException.class )
	public void route_negative() throws ConfigurationException, NamingException, JMSException, ActionProcessingException
	{
		ConfigTree tree = createConfigTree();
		JMSRouter router = new MockJMSRouter(tree);
		router.route( "test" );
	}
	
	@Test
	public void setJMSReplyTo_queue() throws JMSException, URISyntaxException, ConfigurationException, NamingException, ConnectionException
	{
		final String queueName = "testQueue";
		
		TextMessageImpl jmsMessage = new TextMessageImpl();
		
		JMSEpr jmsEpr = new JMSEpr( JMSEpr.QUEUE_TYPE, queueName , "ConnectionFactory" );
		msg.getHeader().getCall().setReplyTo( jmsEpr );
		JMSRouter router = new JMSRouter( createConfigTree() );
		router.setJMSReplyTo( jmsMessage,  msg );
		
		Destination replyTo = jmsMessage.getJMSReplyTo();
		assertTrue( replyTo instanceof Queue );
		Queue replyToQueue = (Queue) replyTo;
		assertEquals( queueName , replyToQueue.getQueueName() );
		
	}
	
	@Test
	@Ignore
	public void setJMSReplyTo_topic() throws JMSException, URISyntaxException, ConfigurationException, NamingException, ConnectionException
	{
		final String queueName = "testTopic";
		
		TextMessageImpl jmsMessage = new TextMessageImpl();
		
		JMSEpr jmsEpr = new JMSEpr( JMSEpr.TOPIC_TYPE, queueName , "ConnectionFactory" );
		msg.getHeader().getCall().setReplyTo( jmsEpr );
		JMSRouter router = new JMSRouter( createConfigTree() );
		router.setJMSReplyTo( jmsMessage,  msg );
		
		Destination replyTo = jmsMessage.getJMSReplyTo();
		assertTrue( replyTo instanceof Topic );
		Topic replyToTopic = (Topic) replyTo;
		assertEquals( queueName , replyToTopic.getTopicName() );
	}
	
	/*
	 * Not quite sure that this is supposed to be testing. 
	 * Can we remove this test? /Daniel
	 */
	@Test
	@Ignore
	public void testRouter () throws Exception
	{
		boolean exception = false;
		
		Message msg = MessageFactory.getInstance().getMessage();
		
		msg.getBody().add("hello world".getBytes());
		JMSRouter router = new JMSRouter( createConfigTree() );
		try
		{
			router.route(msg);
		}
		catch (ActionProcessingException ex)
		{
			exception = true;
		}
		
		if (!exception)
			fail();
		
		router.unwrap = true;
		
		try
		{
			msg.getBody().add("hello world");
			router.process(msg);
		}
		catch (ActionProcessingException ex)
		{
			exception = true;
		}

		if (!exception)
			fail();
		
		router.getErrorNotification(null);
		router.getOkNotification(null);
	}
	
	@Test
	public void construct_with_default_persitent_attribute() throws ConfigurationException, NamingException, JMSException
	{
		ConfigTree config = createConfigTree();
		JMSRouter router = new JMSRouter( config );
		assertTrue( router.isDeliveryModePersistent() );
	}
	
	@Test
	public void construct_with_persitent_attribute() throws ConfigurationException, NamingException, JMSException
	{
		ConfigTree config = createConfigTree();
		config.setAttribute( JMSRouter.PERSISTENT_ATTR, "false" );
		JMSRouter router = new JMSRouter( config );
		
		assertFalse ( router.isDeliveryModePersistent() );
	}
	
	@Test
	public void construct_with_default_priority_attribute() throws ConfigurationException, NamingException, JMSException
	{
		ConfigTree config = createConfigTree();
		JMSRouter router = new JMSRouter( config );
		assertEquals( javax.jms.Message.DEFAULT_PRIORITY, router.getPriority() );
	}
	
	@Test
	public void construct_with_priority_attribute() throws ConfigurationException, NamingException, JMSException
	{
		final int expectedPriority = 9;
		ConfigTree config = createConfigTree();
		config.setAttribute( JMSRouter.PRIORITY_ATTR, String.valueOf( expectedPriority ) );
		JMSRouter router = new JMSRouter( config );
		
		assertEquals ( expectedPriority, router.getPriority() );
	}
	
	@Test
	public void construct_with_default_time_to_live_attribute() throws ConfigurationException, NamingException, JMSException
	{
		ConfigTree config = createConfigTree();
		JMSRouter router = new JMSRouter( config );
		assertEquals( javax.jms.Message.DEFAULT_TIME_TO_LIVE, router.getTimeToLive() );
	}
	
	@Test
	public void construct_with_time_to_live_attribute() throws ConfigurationException, NamingException, JMSException
	{
		final long ttl = 6000l;
		ConfigTree config = createConfigTree();
		config.setAttribute( JMSRouter.TIME_TO_LIVE_ATTR, String.valueOf( ttl ) );
		JMSRouter router = new JMSRouter( config );
		
		assertEquals ( ttl, router.getTimeToLive() );
	}
	
	private void assertProcessContract( final String messageID, final Message msg, JMSRouter router ) throws ActionProcessingException, JMSException
	{
		final Message message = router.process ( msg );
		assertNull ( "Routers process should return null",  message );
		final javax.jms.Message jmsMessage = ((MockJMSRouter) router).getJmsMessage();
		assertEquals ( jmsMessage.getJMSCorrelationID(), messageID );
	}
	
	private static class MockJMSRouter extends JMSRouter
	{
		@SuppressWarnings ( "unused" )
		private Logger log = Logger.getLogger( MockJMSRouter.class );
		
		private javax.jms.Message jmsMessage;

		public MockJMSRouter(ConfigTree propertiesTree) throws ConfigurationException, NamingException, JMSException
		{
			super( propertiesTree );
        }

		@Override
		protected void createQueueSetup( String queueName ) throws ConfigurationException { }

		@Override
		protected void send( javax.jms.Message jmsMessage ) throws JMSException
		{
			this.jmsMessage = jmsMessage;
		}

		public javax.jms.Message getJmsMessage()
		{
			return jmsMessage;
		}
		
		@Override
		protected javax.jms.Message createJMSMessageWithObjectType( Object objectFromBody ) throws JMSException
		{
			TextMessageImpl textMessage = new TextMessageImpl();
			textMessage.setText( new String ((byte[])objectFromBody) );
			return textMessage;
		}

		@Override
		protected javax.jms.Message createObjectMessage( Object message ) throws JMSException
		{
			ObjectMessageImpl impl = new ObjectMessageImpl();
			impl.setObject( (Serializable) message );
			return impl;
		}
	}
	
	private Message createESBMessageObject( final String messageID, final String body) throws URISyntaxException
	{
		Message msg = MessageFactory.getInstance().getMessage();
		msg.getHeader().getCall().setMessageID( new URI ( "1234-junittest" ) );
		msg.getBody().add(body.getBytes());
		return msg;
	}
	
	private static ConfigTree createConfigTree()
	{
		ConfigTree tree = new ConfigTree("test");
		tree.setAttribute("jndiName", "/queue/A");
		return tree;
	}
	
	/*
	 * Just here to help Ant to find annotated test.
	 */
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( JmsRouterIntegrationTest.class );
	}	
	
}
