/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.converters;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.actions.TestBean;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Ignore;
import org.junit.Test;

import com.thoughtworks.xstream.XStream;

/**
 * XStreamToObject unit tests.
 * @author daniel Marchant
 * @author Daniel Bevenius
 * @since Version 4.0
 */
public class XStreamToObjectUnitTest {
	
	private Logger log = Logger.getLogger( XStreamToObjectUnitTest.class );
	
	@Test
    public void test_default() throws ActionProcessingException {
        ConfigTree config = new ConfigTree("config");

        config.setAttribute("incoming-type",TestBean.class.getName());
        
        XStreamToObject xstreamToObject = new XStreamToObject(config);

        Message oMsg = MessageFactory.getInstance().getMessage();

        String msg = "<" + TestBean.class.getSimpleName() + "> <name>Tom</name><phone>1234</phone></" + TestBean.class.getSimpleName() + ">";
        
        oMsg.getBody().add(msg);
        oMsg = xstreamToObject.process(oMsg);
       
        TestBean bean = (TestBean)oMsg.getBody().get();
  
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
         
    }

	@Test
    public void test_with_package() throws ActionProcessingException {
        ConfigTree config = new ConfigTree("config");

        config.setAttribute("exclude-package", "false");
        config.setAttribute("incoming-type", TestBean.class.getName());
        XStreamToObject xstreamToObject = new XStreamToObject(config);

        Message oMsg = MessageFactory.getInstance().getMessage();
        
        String msg = "<" + TestBean.class.getName() + "> <name>Tom</name><phone>1234</phone></"+ TestBean.class.getName() +">";
        
        oMsg.getBody().add(msg);
        oMsg = xstreamToObject.process(oMsg);
       
        TestBean bean = (TestBean)oMsg.getBody().get();
  
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
        
    }

	@Test
    public void test_with_alias() throws ActionProcessingException {
        ConfigTree config = new ConfigTree("config");

        config.setAttribute("class-alias", "TomsClass");
        config.setAttribute("exclude-package", "false");
        config.setAttribute("incoming-type", TestBean.class.getName());
        
        XStreamToObject xstreamToObject = new XStreamToObject(config);

        Message oMsg = MessageFactory.getInstance().getMessage();
        
        String msg = "<TomsClass><name>Tom</name><phone>1234</phone></TomsClass>";
        
        oMsg.getBody().add(msg);
        oMsg = xstreamToObject.process(oMsg);
       
        TestBean bean = (TestBean)oMsg.getBody().get();
  
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
    }
	
	@Test
    public void getAliases()
    {
    	ConfigTree configTree = getConfigTreeWithAliases("TomsClass", false, TestBean.class, null );
        XStreamToObject xstreamToObject = new XStreamToObject( configTree );
        
        Map<String, String> aliases = xstreamToObject.getAliases( configTree );
        
        assertAliases( aliases );
    }
    
	@Test
    public void addAliases() throws ActionProcessingException
    {
    	ConfigTree configTree = getConfigTreeWithAliases("TomsClass", false, TestBean.class, null );
        XStreamToObject xstreamToObject = new XStreamToObject( configTree );
        
        Map<String, String> aliases = xstreamToObject.getAliases( configTree );
        xstreamToObject.addAliases( aliases, new XStream() );
        
        assertAliases( aliases );
    }
    
	@Test
    public void addAliasesNegative() throws ActionProcessingException
    {
    	ConfigTree configTree = getConfigTree( "TomsClass", false, TestBean.class, null );
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass", false, TestBean.class, null );
        
        Map<String, String> aliases = xstreamToObject.getAliases( configTree );
        xstreamToObject.addAliases( aliases, new XStream() );
        
    	assertEquals( "Aliases map should have been empty!", 0,  aliases.size() );
    }
    
	@Test
    public void test_with_multiple_aliases() throws ActionProcessingException {
    	
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass", false, TestBean.class, null );

        Message oMsg = MessageFactory.getInstance().getMessage();
        
        String msg = "<TomsClass><name>Tom</name><phone>1234</phone></TomsClass>";
        
        oMsg.getBody().add(msg);
        oMsg = xstreamToObject.process(oMsg);
       
        TestBean bean = (TestBean)oMsg.getBody().get();
  
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
    }
    
	@Test
    public void test_setRootNodeNegative() throws ActionProcessingException
    {
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass", false, TestBean.class, null );
        assertEquals ( null , xstreamToObject.getRootNodeName() );
    }
    
	@Test
    public void fromXmlToObject_with_rootNode() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/TomsClass";
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass", false, TestBean.class, rootNodeName );
        
        String xml = "<someNode><TomsClass><name>Tom</name><phone>1234</phone></TomsClass></someNode>";
        Object obj = new TestBean();
        
        xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof TestBean );
        assertEquals("1234", ((TestBean)obj).getPhone());
    }
    
	@Test
    public void fromXmlToObject_with_rootNode_as_second_child_node() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/TomsClass2";
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass2", false,TestBean2.class, rootNodeName );
        
        String xml = "<someNode><TomsClass><name>Tom</name><phone>1234</phone></TomsClass><TomsClass2><name>Daniel</name></TomsClass2></someNode>";
        Object obj = new TestBean2();
        
        xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof TestBean2 );
        assertEquals("Daniel", ((TestBean2)obj).getName());
    }
	
	@Test
    public void fromXmlToObject_with_rootNode_as_second_child_node_and_sibling() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/TomsClass2[2]";
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass2", false,TestBean2.class, rootNodeName );
        
        String xml = "<someNode><TomsClass><name>Tom</name><phone>1234</phone></TomsClass><TomsClass2><name>Daniel</name></TomsClass2><TomsClass2><name>Bevenius</name></TomsClass2></someNode>";
        Object obj = new TestBean2();
        
        xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof TestBean2 );
        assertEquals("Bevenius", ((TestBean2)obj).getName());
    }
	
	@Test
    public void fromXmlToObject_with_collection() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/list";
    	ConfigTree configTree = getConfigTree( "list", true, ArrayList.class, rootNodeName);
        ConfigTree alias1 = new ConfigTree( "alias", configTree);
    	alias1.setAttribute( "name", "TomsClass");
    	alias1.setAttribute( "class", TestBean.class.getName());
        XStreamToObject xstreamToObject = new XStreamToObject( configTree );
        
        String xml = "<someNode><list><TomsClass><name>Tom</name></TomsClass><TomsClass><name>Daniel</name></TomsClass></list></someNode>";
        
        Object obj = new ArrayList<TestBean>();
        obj = xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( "Object was not an instance of ArrayList", obj instanceof ArrayList );
        ArrayList<TestBean> nodes = (ArrayList<TestBean>) obj;
        assertEquals( 2, nodes.size() );
        
    }
    
    private XStreamToObject createXStreamObject( String classAlias, boolean exludePackage, Class incomingType, String rootNodeName)
    {
    	ConfigTree configTree = getConfigTree( classAlias, exludePackage, incomingType, rootNodeName);
        return new XStreamToObject( configTree );
    }
    
    private void assertAliases( Map<String, String> aliases )
    {
    	assertNotNull( "Aliases map should not have been null!",  aliases );
        assertEquals( TestBean.class.getName(), aliases.get( "aliasName1" ) );
        assertEquals( TestBean.class.getName(), aliases.get( "aliasName2" ) );
    }
    
    private ConfigTree getConfigTree(String classAlias, boolean excludePackage, Class incomingType, String rootNodeName)
	{
		ConfigTree configTree = new ConfigTree ("test");
    	configTree.setAttribute( "class-alias", classAlias);
    	configTree.setAttribute( "exclude-package", String.valueOf( excludePackage ));
    	configTree.setAttribute( "incoming-type", incomingType.getName());
    	configTree.setAttribute( "root-node", rootNodeName );
		return configTree;
	}

    private ConfigTree getConfigTreeWithAliases(String classAlias, boolean excludePackage, Class incomingType, String rootNodeName)
	{
		ConfigTree configTree = getConfigTree( classAlias,false, incomingType, rootNodeName );
		
    	ConfigTree alias1 = new ConfigTree( "alias", configTree);
    	alias1.setAttribute( "name", "aliasName1");
    	alias1.setAttribute( "class", TestBean.class.getName());
    	
    	ConfigTree alias2 = new ConfigTree( "alias", configTree);
    	alias2.setAttribute( "name", "aliasName2");
    	alias2.setAttribute( "class", TestBean.class.getName());
    	
		return configTree;
	}
    
    /**
     * An abstract class for testing
     */
    private static abstract class AbstractTestBean {     }

    /**
     * Just a bean for testing
     */
	private static class TestBean2 extends AbstractTestBean {
		private String name;

		public String getName()
		{
			return name;
		}

		public void setName( String name )
		{
			this.name = name;
		}
		
	}
	
	/**
	 * Just here to get Ant to find annotated test.
	 */
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( XStreamToObjectUnitTest.class);
	}
	
}	
	