/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.UUID;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.testutils.FileUtil;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Test the Aggregator class
 *
 * @author <a href="mailto:kurt.stam@jboss.com">Kurt Stam</a>
 */


public class AggregatorUnitTest
{
	static Logger logger = Logger.getLogger(AggregatorUnitTest.class);
    static Message message = null;
    
    private static EPR epr1;
    private static EPR epr2;
    private static EPR epr3;
    private static MockCourier courier1;
    private static MockCourier courier2;
    private static MockCourier courier3;
    
    private static ConfigTree[] actions;
	
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(AggregatorUnitTest.class);
    }
    
    @BeforeClass
    public static void before() throws Exception {
        MockCourierFactory.install();
        MockRegistry.install();

        epr1 = new EPR(new URI("test1"));
        epr2 = new EPR(new URI("test2"));
        epr3 = new EPR(new URI("DLS"));
        courier1 = new MockCourier(true);
        courier2 = new MockCourier(true);
        courier3 = new MockCourier(true);

        MockRegistry.register("test", "java", epr1, courier1);
        MockRegistry.register("test", "xml", epr2, courier2);
        MockRegistry.register("test", "aggregator", epr3, courier3);
 
        message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
        message.getBody().add(("Hello Aggregator"));
        
        InputStream in = ClassUtil.getResourceAsStream("AggregatorUnitTest.xml", AggregatorUnitTest.class);
        String xml = FileUtil.readStream(in);
        actions = ConfigTree.fromXml(xml).getChildren("action");
    }
    
    @Test
	public void aggregateThreeMessages()
    {
		try {
            Aggregator aggregator = new Aggregator(actions[0]);
            int recipientCount=3;
            String uuId = UUID.randomUUID().toString();
            long timestamp = System.currentTimeMillis();
            ArrayList<String> aggregatorTags = new ArrayList<String>();
            
            for(int i = 0; i < recipientCount; i++) 
            {
                String tag = uuId + ":" + (i + 1) + ":" + recipientCount + ":" + timestamp;
                aggregatorTags.add(tag);
                message.getProperties().setProperty(Aggregator.AGGEGRATOR_TAG, aggregatorTags);
                Message responseMessage = aggregator.process(message);
                if (i<recipientCount-1) {
                    assertNull(responseMessage);
                } else {
                    assertNotNull(responseMessage);
                }
            }
            
		} catch (Exception e) {
			e.printStackTrace();
			assertTrue(false);
		}
    }
    
    @Test
	public void aggregateWithoutTimeout()
    {
		try {
			InputStream in = ClassUtil.getResourceAsStream("OtherAggregatorUnitTest.xml", AggregatorUnitTest.class);
	        String xml = FileUtil.readStream(in);
	        ConfigTree[] acts = ConfigTree.fromXml(xml).getChildren("action");
	        
            Aggregator aggregator = new Aggregator(acts[0]);
		} catch (Exception e) {
			assertTrue(false);
		}
    }
    
    @Test
    public void aggregateTimeoutTest1()
    {
        try {
            Aggregator aggregator = new Aggregator(actions[1]);
            int recipientCount=3;
            String uuId = UUID.randomUUID().toString();
            long timestamp = System.currentTimeMillis();
            ArrayList<String> aggregatorTags = new ArrayList<String>();
            
            for(int i = 0; i < recipientCount; i++) 
            {
                String tag = uuId + ":" + (i + 1) + ":" + recipientCount + ":" + timestamp;
                aggregatorTags.add(tag);
                message.getProperties().setProperty(Aggregator.AGGEGRATOR_TAG, aggregatorTags);
                
                try {
                    Thread.sleep(200);
                } catch (InterruptedException e) {}
                
                Message responseMessage = aggregator.process(message);
                
                //all message should be expired
                assertNull(responseMessage);
                
            }
            
        } catch (Exception e) {
            assertTrue(false);
        }
    }
    
}