/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.embedded.EmbeddableException;
import org.jboss.internal.soa.esb.util.embedded.ftp.FtpTestUtil;
import org.jboss.internal.soa.esb.util.embedded.ftp.NoConfigFileFtpServer;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.util.RemoteFileSystem;
import org.jboss.soa.esb.util.RemoteFileSystemException;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Unit test for EdtFtpImpl that uses an embedded ftp server
 * 
 * @author Daniel Bevenius
 * 
 */
public class EdtFtpImplUnitTest 
{
	private static Logger log = Logger.getLogger( EdtFtpImplUnitTest.class );
	
	/* EmbeddedFtp Server */
	private static NoConfigFileFtpServer ftpServer;

	/* Instance of class under test */
	private static EdtFtpImpl edtFtpImpl;
	
	private static String remoteInputDirName;
	private static String remoteUploadDirName;
	private static final String INPUT_SUFFIX = ".txt";
	private static final String RENAMED_SUFFIX = ".renamed";
	
	/* Content for created test files */
	private static final String TEST_FILE_CONTENT = EdtFtpImplUnitTest.class .getName() + " junit ftp test";
	private File testFile;
	private File renamedFile;

	@BeforeClass
	public static void classSetup() throws EmbeddableException, ConfigurationException, RemoteFileSystemException, MalformedURLException, URISyntaxException
	{
		ftpServer = new NoConfigFileFtpServer();
		ftpServer.setPort( 2221 );
		ftpServer.start();

		remoteInputDirName = "/" + ftpServer.getLocalInputDir().getName();
		remoteUploadDirName = "/" + ftpServer.getLocalUploadDir().getName();
		edtFtpImpl = new EdtFtpImpl( createConfigTree(), true );
	}

	@AfterClass
	public static void classTearDown() throws EmbeddableException
	{
		try
		{
			ftpServer.stop();
		} 
		catch (Exception e)
		{
			log.warn( e.getMessage() );
		}
		
		if  ( !FtpTestUtil.deleteDir( ftpServer.getFtpServerDir() ) )
		{
			log.warn( "Could not delete " +  ftpServer.getFtpServerDir() ) ;
		}
	}

	@Before
	public void setUp()
	{
		testFile = FtpTestUtil.createTestFile( ftpServer.getLocalInputDir(), getClass().getName() + INPUT_SUFFIX, TEST_FILE_CONTENT );
	}
	
	@After
	public void tearDown()
	{
		FtpTestUtil.deleteFile( testFile );
		FtpTestUtil.deleteFile( renamedFile );
	}
	
	@Test
	public void construtor_FTPEpr() 
	{
		try
		{
			new EdtFtpImpl( createFTPEpr(), true );
		}
		catch ( Exception e )
		{
			fail ( e.getMessage() );
		}
	}
	
	@Test
	public void getRemoteDir() throws RemoteFileSystemException
	{
		String oldRemoteDir =  remoteInputDirName;
		try
		{
			edtFtpImpl.setRemoteDir( remoteUploadDirName );
			assertEquals( "The remote dir was not changed.", remoteUploadDirName, edtFtpImpl.getRemoteDir() );
		} 
		finally
		{
			try { edtFtpImpl.setRemoteDir( oldRemoteDir ); } catch (Exception e) { fail( e.getMessage() ); }
		}
	}

	@Test
	public void setRemoteDir() throws RemoteFileSystemException
	{
		String oldRemoteDir = remoteInputDirName;
		try
		{
			edtFtpImpl.setRemoteDir( remoteUploadDirName );

			String actualRemoteDir = edtFtpImpl.getRemoteDir();
			assertFalse( "The setRemoteDir method did not change the directory!", oldRemoteDir.equals( actualRemoteDir ) );
		} 
		finally
		{
			try { edtFtpImpl.setRemoteDir( oldRemoteDir ); } catch (Exception e) { fail( e.getMessage() ); }
		}
	}

	@Test
	public void getFileListFromRemoteDir() throws RemoteFileSystemException, IOException
	{
		String[] fileListFromRemoteDir = edtFtpImpl .getFileListFromRemoteDir( INPUT_SUFFIX );
		assertNotNull( fileListFromRemoteDir );
		List<String> fileList = Arrays.asList( fileListFromRemoteDir );
		assertTrue( "The test file was not included in the List! ", fileList.contains( testFile.getName() ) );
	}

	@Test
	public void deleteRemoteFile() throws RemoteFileSystemException
	{
		edtFtpImpl.deleteRemoteFile( testFile.getName() );
		assertFalse( "File was not deleted", testFile.exists() );
	}

	@Test
	public void remoteDelete() throws RemoteFileSystemException
	{
		edtFtpImpl.remoteDelete( testFile );
		assertFalse( testFile.exists() );
	}

	@Test
	public void renameInRemoteDir() throws RemoteFileSystemException
	{
		File from = testFile;
		String toFileName = from.getName() + RENAMED_SUFFIX;
			
		edtFtpImpl.renameInRemoteDir( from.getName(), toFileName );

		renamedFile = new File( ftpServer.getLocalInputDir(), toFileName );
		assertFalse( "The file was not removed from the filesystem", from .exists() );
		assertTrue( "The named file does not exist", renamedFile.exists() );
	}

	@Test
	public void remoteRename_To_Different_RemoteDir() throws RemoteFileSystemException
	{
		File from = createAbsoluteFromFile();
		File to = createAbsoluteToFileUploadDir();

		edtFtpImpl.remoteRename( from, to );

		renamedFile = new File( ftpServer.getLocalUploadDir(), to.getName() );
		assertFalse( "The file was not removed from the filesystem", from .exists() );
		assertTrue( renamedFile + " was not found in dir " + ftpServer.getLocalUploadDir(), renamedFile.exists() );
	}
	
	@Test
	public void remoteRename_To_Same_RemoteDir() throws RemoteFileSystemException
	{
		File from =  createAbsoluteFromFile();
		File to = createAbsoluteToFile();
			
		edtFtpImpl.remoteRename( from, to );

		renamedFile = new File( ftpServer.getLocalInputDir(), testFile.getName() + RENAMED_SUFFIX );
		assertFalse( testFile.exists() );
		assertTrue( renamedFile.exists() );
	}
	
	@Test
	public void downloadFile() throws RemoteFileSystemException, IOException
	{
		File localFile = null;
		File localDownloadedlFile = null;
		String testFileName = "downloadFileTestFile.txt";
		try
		{
			localFile = FtpTestUtil.createTestFile( ftpServer .getLocalInputDir(), testFileName, TEST_FILE_CONTENT );
			String remoteFileName = "/input/" + localFile.getName();
			String downloadFileName = testFileName + ".downloaded";

			edtFtpImpl.downloadFile( remoteFileName, downloadFileName );

			localDownloadedlFile = new File( ftpServer.getRootDir(), downloadFileName );
			assertTrue( "The remote file should not have been removed.", localFile.exists() );
			assertTrue( "File was not downloaded ", localDownloadedlFile .exists() );
		} 
		finally
		{
			FtpTestUtil.deleteFile( localDownloadedlFile );
			FtpTestUtil.deleteFile( localFile );
		}
	}

	@Test
	public void uploadFile() throws RemoteFileSystemException
	{
		File localFile = null;
		File remoteFile = null;
		String testFileName = "uploadFileTestFile.txt";
		try
		{
			localFile = FtpTestUtil.createTestFile(  ftpServer.getLocalInputDir(),  testFileName, TEST_FILE_CONTENT );
			
			String renameTo = testFileName + RENAMED_SUFFIX;
			edtFtpImpl.uploadFile( localFile, renameTo );
			remoteFile = new File ( ftpServer.getLocalInputDir(), renameTo );
			
			assertTrue( "The file was not uploaded succesfully." , remoteFile.exists() );
		} 
		finally
		{
			FtpTestUtil.deleteFile( remoteFile );
			FtpTestUtil.deleteFile( localFile );
		}
	}

	private static ConfigTree createConfigTree()
	{
		ConfigTree configTree = new ConfigTree( "junitEdtFtpImplTest" );
		try
		{
			configTree.setAttribute( FileEpr.URL_TAG, ftpServer.getURL().toString() ) ;
		}
		catch ( MalformedURLException e )
		{
			log.error(  e  );
			fail ( e.getMessage() );
		}
		configTree.setAttribute( RemoteFileSystem.PARMS_FTP_SERVER, ftpServer .getHost() );
		configTree.setAttribute( RemoteFileSystem.PARMS_USER, ftpServer .getUserName() );
		configTree.setAttribute( RemoteFileSystem.PARMS_PASSWD, ftpServer .getPassword() );
		configTree.setAttribute( RemoteFileSystem.PARMS_REMOTE_DIR, remoteInputDirName );
		configTree.setAttribute( RemoteFileSystem.PARMS_PORT, Integer .toString( ftpServer.getPort() ) );
		configTree.setAttribute( RemoteFileSystem.PARMS_LOCAL_DIR, ftpServer .getRootDir() );
		configTree.setAttribute( RemoteFileSystem.PARMS_ASCII, Boolean .toString( false ) );
		configTree.setAttribute( RemoteFileSystem.PARMS_PASSIVE, Boolean .toString( false ) );
		return configTree;
	}
	
	private static FTPEpr createFTPEpr() throws MalformedURLException, URISyntaxException
	{
		FTPEpr epr = new FTPEpr ( new EPR() );
		epr.setURL( ftpServer.getURL() );
		epr.setUserName( ftpServer.getUserName() );
		epr.setErrorDirectory( ftpServer.getErrorDirName() );
		epr.setInputSuffix( INPUT_SUFFIX );
		epr.setPassword( ftpServer.getPassword() );
		epr.setPostDirectory( ftpServer.getUploadDirName() );
		epr.setWorkSuffix( ".work" );
		return epr;
	}
	
	private File createAbsoluteFromFile()
	{
		return new File ( remoteInputDirName + "/" + testFile.getName() );
	}
	
	private File createAbsoluteToFile()
	{
		return new File ( remoteInputDirName + "/" + testFile.getName() + RENAMED_SUFFIX );
	}
	
	private File createAbsoluteToFileUploadDir()
	{
		return new File ( remoteUploadDirName + "/" +  testFile.getName() + RENAMED_SUFFIX );
	}

	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( EdtFtpImplUnitTest.class);
	}
	
}
