/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.schedule;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.util.ClassUtil;
import org.quartz.*;
import org.quartz.impl.StdSchedulerFactory;
import org.apache.log4j.Logger;

import java.util.*;
import java.text.ParseException;
import java.io.IOException;
import java.io.InputStream;

/**
 * Schedule Provider.
 * <p/>
 * Manages all of the configured scheduled jobs and their listeners.
 *
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ScheduleProvider {

    private static Logger logger = Logger.getLogger(ScheduleProvider.class);

    /* Quartz scheduler instance */
    private Scheduler scheduler;
    private List<String> scheduledJobs = new ArrayList<String>();
    private List<Schedule> schedules;
    
    private final ClassLoader tcc ;
    private static final String JBOSS_ESB = "JBossESB";

    public ScheduleProvider(Properties properties, List<Schedule> schedules) throws ConfigurationException {
        try {
            Properties defaultProperties = new Properties();
            InputStream quartzProperties = ClassUtil.getResourceAsStream("quartz.properties", ScheduleProvider.class);

            if(quartzProperties != null) {
                defaultProperties.load(quartzProperties);
            }
            if(properties != null && !properties.isEmpty()) {
                // Overlay the defaults with the configs on the schedule config...
                defaultProperties.putAll(properties);
            }
            if(!defaultProperties.isEmpty()) {
                scheduler = new StdSchedulerFactory(defaultProperties).getScheduler();
            } else {
                scheduler = new StdSchedulerFactory().getScheduler();
            }
            tcc = Thread.currentThread().getContextClassLoader() ;
        } catch (SchedulerException e) {
            throw new ConfigurationException("Unable to create Scheduler instance.", e);
        } catch (IOException e) {
            throw new ConfigurationException("Unable to create Scheduler instance. Failed to read", e);
        }
        this.schedules = schedules;
    }

    public void addListener(ScheduledEventListener listener, String scheduleId) throws SchedulingException, ConfigurationException {
        Schedule schedule = getSchedule(scheduleId);
        Trigger trigger;

        if(schedule == null) {
            throw new ConfigurationException("Unknown schedule '" + scheduleId + "' referenced in listener configuration.");
        }

        if(schedule instanceof SimpleSchedule) {
            SimpleSchedule simpleSchedule = (SimpleSchedule) schedule;
            if(simpleSchedule.getExecCount() == -1) {
                trigger = TriggerUtils.makeSecondlyTrigger((int) simpleSchedule.getFrequency());
            } else if(simpleSchedule.getExecCount() == 0) {
                logger.warn("<simple-schedule> '" + scheduleId + "' has an execCount of 0 configured.  This schedule will not fire!");
                return;
            } else {
                // So the repeatCount = execCount - 1 ....
                trigger = TriggerUtils.makeSecondlyTrigger((int) simpleSchedule.getFrequency(), (simpleSchedule.getExecCount() - 1));
            }
            trigger.setName(simpleSchedule.getScheduleid());
        } else {
            String cronExpression = ((CronSchedule)schedule).getCronExpression();
            try {
                trigger = new CronTrigger(scheduleId, JBOSS_ESB, cronExpression);
            } catch (ParseException e) {
                throw new ConfigurationException("Invalid CRON expression '" + cronExpression + "' on schedule '" + scheduleId + "'.", e);
            }
        }

        // Set the start and end times, if configured....
        if(schedule.getStartDate() != null) {
            trigger.setStartTime(schedule.getStartDate().getTime());
        } else {
            trigger.setStartTime(new Date());
        }
        if(schedule.getEndDate() != null) {
            trigger.setEndTime(schedule.getEndDate().getTime());
        }

        addListener(listener, trigger);
    }

    public void addListener(ScheduledEventListener listener, long scheduleFrequency) throws SchedulingException {
        Trigger trigger = TriggerUtils.makeSecondlyTrigger((int) scheduleFrequency);
        String name = (listener.toString() + scheduleFrequency);

        trigger.setName(name);
        trigger.setStartTime(new Date());
        addListener(listener, trigger);
    }

    private static volatile int nameDelta = 1; 
    private void addListener(ScheduledEventListener listener, Trigger trigger) throws SchedulingException {
        JobDataMap jobDataMap = new JobDataMap();
        JobDetail jobDetail;

        String name = trigger.getName();

        // this is just to make sure they're unique - i.e. so as 1+
        // "things" can listen to the same schedule...
        // This is not thread safe!
        name += ("-" + nameDelta++);
        trigger.setName(name);

        jobDetail = new JobDetail(name, JBOSS_ESB, ESBScheduledJob.class);
        jobDataMap.put(ScheduledEventListener.class.getName(), listener);
        jobDataMap.put(ClassLoader.class.getName(), tcc);
        jobDetail.setJobDataMap(jobDataMap);
        try {
            scheduler.scheduleJob(jobDetail, trigger);
            scheduledJobs.add(name);
        } catch (SchedulerException e) {
            throw new SchedulingException("Failed to schedule job.", e);
        }
    }

    private Schedule getSchedule(String scheduleId) {
        for(Schedule schedule : schedules) {
            if(schedule.getScheduleid().equals(scheduleId.trim())) {
                return schedule;
            }
        }

        return null;
    }

    /**
     * Start the scheduler.
     * @throws SchedulingException Start failed.
     */
    public void start() throws SchedulingException {
        try {
            scheduler.start();
        } catch (SchedulerException e) {
            throw new SchedulingException("Failed to start scheduling.", e);
        }
    }

    /**
     * Standby the scheduler.
     * <p/>
     * Restart the scheduler through the {@link #start()} method.
     *
     * @throws SchedulingException Standby failed.
     */
    public void standby() throws SchedulingException {
        try {
            scheduler.standby();
        } catch (SchedulerException e) {
            throw new SchedulingException("Failed to standby scheduling.", e);
        }
    }

    /**
     * Stop the scheduler.
     * @throws SchedulingException Stop failed.
     */
    public void stop() throws SchedulingException {
        try {
            for(String jobName : scheduledJobs) {
                if(!scheduler.deleteJob(jobName, JBOSS_ESB)) {
                    logger.info("Failed to delete scheduled Job '" + jobName + "' from job group '" + JBOSS_ESB + "'.  Job run may have already completed.");
                }
            }
        } catch (SchedulerException e) {
            throw new SchedulingException("Failed to shutdown scheduling.", e);
        }
    }

    public static class ESBScheduledJob implements StatefulJob {
        public void execute(JobExecutionContext jobExecutionContext) throws JobExecutionException {
            final JobDataMap jobDataMap = jobExecutionContext.getJobDetail().getJobDataMap() ;
            ScheduledEventListener listener = (ScheduledEventListener) jobDataMap.get(ScheduledEventListener.class.getName());
            final ClassLoader tcc = (ClassLoader)jobDataMap.get(ClassLoader.class.getName());
            
            final Thread thread = Thread.currentThread() ;
            final ClassLoader currentClassLoader = thread.getContextClassLoader() ;
            thread.setContextClassLoader(tcc) ;
            try {
                listener.onSchedule();
            } catch (SchedulingException e) {
                JobExecutionException jobException = new JobExecutionException("Scheduling exception on " + jobExecutionContext.getJobDetail().getName());
                jobException.initCause(e);
                throw jobException;
            } catch (final Throwable th) {
                JobExecutionException jobException = new JobExecutionException("Unexpected exception on " + jobExecutionContext.getJobDetail().getName());
                jobException.initCause(th);
                throw jobException;
            } finally {
                thread.setContextClassLoader(currentClassLoader) ;
            }
        }
    }
}
