/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.notification;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageType;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;

/**
 * Write the notification contents into a list of files specified in the
 * constructor time parameters.
 * <p>
 * Description: The constructor searches for all child elements of the
 * ConfigTree argument having "file" as element name, that will be used to store
 * (or append) the contents of the argument to the sendNotification(Object)
 * method, to each and every one of the files contained in the list
 * </p>
 * <p>
 * Author: Heuristica - Buenos Aires - Argentina
 * </p>
 * 
 * @version 1.0
 */
public class NotifyFiles extends NotificationTarget
{
	private Logger log = Logger.getLogger( NotifyFiles.class );
	
	/**
	 * Mnemonic for the child element name that hold the files to write ("file")
	 */
	public static final String CHILD_FILE = "file";

	/**
	 * Attribute name that will be interpreted as the URI of the file to write
	 * to
	 */
	public static final String ATT_URI = "URI";

	/**
	 * Attribute name of indicator to append to an existing file
	 * 
	 * @see Boolean#valueOf(String)
	 */
	public static final String ATT_APPEND = "append";

	/**
	 * The NotificationFile[] that holds the output file list.
	 */
	protected NotificationFile[] m_oaOutF;
    private MessagePayloadProxy payloadProxy;

    /**
	 * Instantiate a NotifyFiles object according to contents of &lt;arg 1&gt;
	 * 
	 * @param configTree
	 *            ConfigTree - Should contain a nonempty set of child elements
	 *            with elementName="file". Each child element must have a "URI"
	 *            attribute and optionally a "append" element
	 */
	public NotifyFiles (ConfigTree configTree)
	{
		super(configTree);
		setFiles(configTree.getChildren(CHILD_FILE));
        payloadProxy = new MessagePayloadProxy(configTree,
                                               new String[] {BytesBody.BYTES_LOCATION},
                                               new String[] {BytesBody.BYTES_LOCATION});

	} // __________________________________

	/**
	 * Sets the value of m_oaOutF[] and m_baAppend[] to contents of each element
	 * of the input argument
	 * 
	 * @param p_oaP
	 *            ConfigTree[] - Each entry must have a "URI" attribute, and can
	 *            have an optional "append" attribute
	 * @see ConfigTree#getAttribute(String)
	 */
	protected void setFiles (ConfigTree[] p_oaP)
	{
		m_oaOutF = new NotificationFile[p_oaP.length];

		for (int i = 0; i < p_oaP.length; i++)
		{
			String fileURIString = p_oaP[i].getAttribute(ATT_URI);
			String append = p_oaP[i].getAttribute(ATT_APPEND);

			if (null == fileURIString)
			{
				throw new IllegalArgumentException(
						"Bad File Notification Configuration: Missing file URI attribute.");
			}

            fileURIString = fileURIString.replace('\\', '/');
            try {
                URI fileURI = new URI(fileURIString);
                
                if("file".equalsIgnoreCase(fileURI.getScheme())) {
                    if(fileURI.getHost() != null) {
                        // Remote dir ref... not supported...
                        throw new IllegalArgumentException("Sorry, the NotifyFiles notifier doesn't support remote directories: '" + fileURIString + "'.  To reference a local dir, add a forward slash character before '" + fileURI.getAuthority() + "'.");
                    }
                    m_oaOutF[i] = new NotificationFile(fileURI, Boolean.valueOf(append));
                } else {
                    // Not a file based URI... don't use the URI object...
                    m_oaOutF[i] = new NotificationFile(fileURIString, Boolean.valueOf(append));
                }
            } catch (URISyntaxException e) {
                m_oaOutF[i] = new NotificationFile(fileURIString, Boolean.valueOf(append));
            }

			// Make sure the parent folder exists...
			File parent = m_oaOutF[i].getAbsoluteFile().getParentFile();
			if (null == parent || !parent.exists())
			{
				throw new IllegalArgumentException(
						"Bad File Notification Configuration: Parent folder for file [" + m_oaOutF[i]
								.getAbsolutePath() + "] doesn't exist.");
			}
		}
	} // __________________________________

	/**
	 * Writes the result of p_o into each one of the File objects contained in
	 * the m_oaOutF array
	 * 
	 * @param message
	 *            Object - This object's toString() results will be written to
	 *            (appended to) each one of the files in m_oaOutF
	 * @see NotifyFiles#setFiles(ConfigTree[])
	 * @see NotifyFiles#m_oaOutF
	 */
	public void sendNotification (Message message) throws NotificationException
	{
		FileOutputStream fileOutStream = null;
        
        final StringBuilder exceptions = new StringBuilder();
		for (NotificationFile notificationFile : m_oaOutF)
		{
		    try
		    {
			fileOutStream = new FileOutputStream(notificationFile,
				notificationFile.append);
			Object obj = payloadProxy.getPayload(message);

			/*
			 * TODO
			 * 
			 * Hmmm, why was this keying off the ESB message type? Change it so
			 * that if it's a byte[] or a String then we'll use the stringNotification.
			 * Otherwise output as a generic Serializable. Slightly different to other
			 * notifiers (except JMSNotifier), but probably the best we can do in the
			 * situation.
			 * 
			 * http://jira.jboss.com/jira/browse/JBESB-1607
			 */
			
			if ((obj instanceof Serializable) && !(obj instanceof byte[]) && !(obj instanceof String))
			{
			    objectNotification(fileOutStream, payloadProxy.getPayload(message));
			}
			else
			{
			    String content=null;
			    if (obj instanceof byte[]) {
				content = new String((byte[]) obj);
			    } else {
				content = obj.toString();
			    }
			    stringNotification(fileOutStream, content);
			}
		    }
		    catch (IOException e)
		    {
			handleException(notificationFile, e, exceptions);
		    }
		    catch (MessageDeliverException e) {
			handleException(notificationFile, e, exceptions);
		    } finally
		    {
			try
			{
			    if ( fileOutStream != null )
				fileOutStream.close();
			}
			catch (IOException eCl)
			{
			    log.error( "IOException while closing fileOutStream: ", eCl );
			}
		    }
		}
		if ( exceptions.length() > 0 )
			throw new NotificationException( exceptions.toString() );
	} // __________________________________

    private void handleException(NotificationFile notificationFile, Exception e, StringBuilder exceptions) {
        final String msg = "[Exception while notifying file : " + notificationFile;
        log.error(msg, e);
        exceptions.append( NotifyUtil.createExceptionErrorString( msg, e ));
    }

    protected void stringNotification (FileOutputStream p_oF, String p_s) throws IOException
	{
		p_oF.write(p_s.getBytes());
		if (!p_s.endsWith("\n")) 
			p_oF.write("\n".getBytes());
	} // __________________________________

	protected void objectNotification (FileOutputStream p_oF, Object p_o) throws IOException
	{
		ObjectOutputStream OS = new ObjectOutputStream(p_oF);
		OS.writeObject(p_o);
	} // __________________________________

	/**
	 * Notification File. <p/> Simply adds the "append" property to the file.
	 * 
	 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
	 */
	private class NotificationFile extends File
	{

		private static final long serialVersionUID = 1L;

		private boolean append = false;

		private NotificationFile (URI fileURI, boolean append)
		{
			super(fileURI);
			this.append = append;
		}

		/**
		 * @param file
		 * @param append
		 */
		public NotificationFile (String file, boolean append)
		{
			super(file);
			this.append = append;
		}
	}
} // ____________________________________________________________________________
