/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.notification;

import javax.mail.MessagingException;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.Email;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.util.Util;

/**
 * This class will send an e-mail using the Email class
 * <p>
 * Description:
 * </p>
 * <p>
 * Author: Heuristica - Buenos Aires - Argentina
 * </p>
 * 
 * @version 1.0
 */
public class NotifyEmail extends NotificationTarget
{
    private MessagePayloadProxy payloadProxy;

    /**
	 * Instantiate a NotifyEmail object using the information contained in
	 * &lt;arg 1&gt;
	 * 
	 * @param configTree
	 *            ConfigTree - See attributes and structure needed for the
	 *            Email() constructor - The MESSAGE attribute will be filled in
	 *            at sendNotification(Serializable) time
	 * @throws ConfigurationException
	 * @see NotifyEmail#sendNotification(Message)
	 */
	public NotifyEmail (ConfigTree configTree) throws ConfigurationException
	{
		super(configTree);

		try
		{
			String sAtt = (String) m_oParms.getAttribute(Email.FROM);
			
			if (null != sAtt) 
				InternetAddress.parse(sAtt);
	
			InternetAddress.parse(m_oParms.getAttribute(Email.SENDTO));
	
			sAtt = (String) m_oParms.getAttribute(Email.COPYTO);
			if (null != sAtt) 
				InternetAddress.parse(sAtt);
		}
		catch (AddressException ex)
		{
			throw new ConfigurationException(ex);
		}
        payloadProxy = new MessagePayloadProxy(configTree,
                                               new String[] {BytesBody.BYTES_LOCATION},
                                               new String[] {BytesBody.BYTES_LOCATION});
	} // __________________________________

	/**
	 * Send an Email using Email() using p_o.toString() to fill in the message
	 * text
	 * 
	 * @param message
	 *            Object - This object's toString() method will supply contents
	 *            of mail message
	 */
	public void sendNotification (Message message) throws NotificationException
	{
		try
		{
            Object obj = payloadProxy.getPayload(message);
            String content;

            if(obj instanceof byte[]) {
                content = new String((byte[]) obj);
            } else {
                content = obj.toString();
            }

			ConfigTree oP = m_oParms.cloneObj();
			String sMsg = oP.getAttribute(Email.MESSAGE);
                        sMsg = ((null == sMsg) ? content : (sMsg + "\n" + content));
			oP.setAttribute(Email.MESSAGE, sMsg);
			sendEmailNotification(oP);
        } catch (MessageDeliverException e) {
            throw new NotificationException(e);
        }
		catch (AddressException e)
		{
			Util.getDefaultLogger(this.getClass()).error("Send Mail Failed", e);
			
			throw new NotificationException(e);
		}
		catch (MessagingException ex)
		{
			throw new NotificationException(ex);
		}
	} // __________________________________

	/**
	 * Send an email notification based on the supplied parameters. <p/> This
	 * method allows overriding for test purposes.
	 * 
	 * @param messageParams
	 *            Message parameters.
	 */
	protected void sendEmailNotification (ConfigTree messageParams)
			throws AddressException, MessagingException
	{

		Email esbMail = new Email();
		esbMail.setSendTo(messageParams.getAttribute(Email.SENDTO));
		esbMail.setFrom(messageParams.getAttribute(Email.FROM));
		esbMail.setCopyTo(messageParams.getAttribute(Email.COPYTO));
		esbMail.setSubject(messageParams.getAttribute(Email.SUBJECT));

		esbMail.setAttachments(messageParams.getTextChildren(Email.ATTACH));
		esbMail.setMessage(messageParams.getAttribute(Email.MESSAGE));

		esbMail.sendMessage();
	}

} // ____________________________________________________________________________
