/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.message.Message;

/**
 * Class providing information for dynamic manipulation of the action processors. 
 * @author kevin
 */
class ActionProcessorMethodInfo
{
    /**
     * The logger for this class.
     */
    private static final Logger LOGGER = Logger.getLogger(ActionProcessorMethodInfo.class) ;
    
    /**
     * The process methods.
     */
    private final Method[] processMethods;
    /**
     * The process success method.
     */
    private final Method processSuccess;
    /**
     * The process exception method.
     */
    private final Method processException ;
    
    /**
     * Construct the action processor method info.
     * @param processMethods The process methods or null if not configured.
     * @param processSuccess The process success method or null if not configured.
     * @param processException The process exception method or null if not configured.
     */
    private ActionProcessorMethodInfo(final Method[] processMethods, final Method processSuccess,
        final Method processException)
    {
        this.processMethods = processMethods ;
        this.processSuccess = processSuccess ;
        this.processException = processException ;
    }

    /**
     * Do we have process method overrides?
     * @return true if there are overrides, false otherwise.
     */
    boolean hasProcessMethods()
    {
        return (processMethods != null) ;
    }
    
    /**
     * Process the methods.
     * @param instance The action processor instance.
     * @param message The initial message.
     * @return The last message.
     * @throws ActionProcessingException for errors during processing.
     */
    Message processMethods(final Object instance, final Message message)
        throws ActionProcessingException
    {
        Message currentMessage = message ;
        if (hasProcessMethods())
        {
            final int numProcessMethods = processMethods.length ;
            for(int count = 0 ; count < numProcessMethods ; count++)
            {
                final Method processMethod = processMethods[count] ;
                try
                {
                    final Object response = processMethod.invoke(instance, currentMessage) ;
                    if (response instanceof Message)
                    {
                        currentMessage = (Message)response ;
                    }
                    else if (response == null)
                    {
                        return null ;
                    }
                    else
                    {
                        throw new ActionProcessingException("Unexpected response type from processor: " + response) ;
                    }
                }
                catch (final IllegalAccessException iae)
                {
                    throw new ActionProcessingException("Illegal access from processor", iae) ;
                }
                catch (final InvocationTargetException ite)
                {
                    final Throwable th = ite.getTargetException() ;
                    if (th instanceof ActionProcessingException)
                    {
                        throw (ActionProcessingException)th ;
                    }
                    else if (th instanceof RuntimeException)
                    {
                        throw (RuntimeException)th ;
                    }
                    else if (th instanceof Error)
                    {
                        throw (Error)th ;
                    }
                    else
                    {
                        throw new ActionProcessingException("Unexpected invocation target exception from processor", th) ;
                    }
                }
            }
        }
        return currentMessage ;
    }

    /**
     * Do we have a process success override?
     * @return true if there is an override, false otherwise.
     */
    boolean hasProcessSuccess()
    {
        return (processSuccess != null) ;
    }
    
    /**
     * Process the success method.
     * @param instance The action processor instance.
     * @param message The initial message.
     */
    void processSuccess(final Object instance, final Message message)
    {
        if (hasProcessSuccess())
        {
            try
            {
                processSuccess.invoke(instance, message) ;
            }
            catch (final IllegalAccessException iae)
            {
                LOGGER.warn("Illegal access from processor", iae) ;
            }
            catch (final InvocationTargetException ite)
            {
                final Throwable th = ite.getTargetException() ;
                LOGGER.warn("Unexpected invocation target exception from processor", th) ;
            }
        }
    }


    /**
     * Do we have a process exception override?
     * @return true if there is an override, false otherwise.
     */
    boolean hasProcessException()
    {
        return (processException != null) ;
    }
    
    /**
     * Process the exception method.
     * @param instance The action processor instance.
     * @param message The initial message.
     * @param ex The exception.
     */
    void processException(final Object instance, final Message message, final Throwable th)
    {
        if (hasProcessException())
        {
            try
            {
                processException.invoke(instance, message, th) ;
            }
            catch (final IllegalAccessException iae)
            {
                LOGGER.warn("Illegal access from processor", iae) ;
            }
            catch (final InvocationTargetException ite)
            {
                LOGGER.warn("Unexpected invocation target exception from processor", ite.getTargetException()) ;
            }
        }
    }

    /**
     * Check to see if any of the default configurations have been overridden.
     * @param actionConfig The action config.
     * @return true if there is an override, false otherwise.
     */
    static boolean checkOverridden(final ConfigTree actionConfig)
    {
        final String processMethod = actionConfig.getAttribute(ListenerTagNames.PROCESS_METHOD_TAG) ;
        if (processMethod == null || ActionPipelineProcessor.PROCESS_METHOD.equals(processMethod))
        {
            final String successMethod = actionConfig.getAttribute(ListenerTagNames.NORMAL_COMPLETION_METHOD_TAG) ;
            if (successMethod == null || ActionPipelineProcessor.PROCESS_SUCCESS_METHOD.equals(successMethod))
            {
                final String exceptionMethod = actionConfig.getAttribute(ListenerTagNames.EXCEPTION_METHOD_TAG) ;
                if (exceptionMethod == null || ActionPipelineProcessor.PROCESS_EXCEPTION_METHOD.equals(exceptionMethod))
                {
                    return false ;
                }
            }
        }
        return true ;
    }

    /**
     * Get the method info for method overrides.
     * @param actionConfig The action config.
     * @param actionClass The action class.
     * @return The action processor method info.
     * @throws ConfigurationException for configuration errors
     */
    static ActionProcessorMethodInfo getMethodInfo(final ConfigTree actionConfig, final Class actionClass)
        throws ConfigurationException
    {
        try
        {
            final String processMethodName = actionConfig.getAttribute(ListenerTagNames.PROCESS_METHOD_TAG, ActionPipelineProcessor.PROCESS_METHOD) ;
            final Method[] processMethods = getMethods(actionClass, processMethodName, Message.class);
            
            final String processSuccessName = actionConfig.getAttribute(ListenerTagNames.NORMAL_COMPLETION_METHOD_TAG) ;
            final Method processSuccess = getMethod(actionClass, processSuccessName, Message.class);
            
            final String processExceptionName = actionConfig.getAttribute(ListenerTagNames.EXCEPTION_METHOD_TAG) ;
            final Method processException = getMethod(actionClass, processExceptionName, Message.class, Throwable.class) ;
            
            return new ActionProcessorMethodInfo(processMethods, processSuccess, processException) ;
        }
        catch (final NoSuchMethodException nsme)
        {
            throw new ConfigurationException("Invalid processor method in configuration", nsme) ;
        }
    }
    
    /**
     * Get the methods specified by the list.
     * @param actionClass The action class
     * @param methodNameList The list of method names
     * @param classes The method signature classes.
     * @return The methods.
     * @throws NoSuchMethodException If a method cannot be located.
     */
    private static Method[] getMethods(final Class actionClass, final String methodNameList, Class ...classes)
        throws NoSuchMethodException
    {
        final String[] methodNames = methodNameList.split(",") ;
        final int numMethodNames = (methodNames == null ? 0 : methodNames.length) ;
        
        final Method[] methods = new Method[numMethodNames] ;
        
        for(int count = 0 ; count < numMethodNames ; count++)
        {
            final String methodName = methodNames[count].trim() ;
            methods[count] = getMethod(actionClass, methodName, classes) ;
        }
        return methods ;
    }
    
    /**
     * Get the specified method.
     * @param actionClass The action class
     * @param methodName The method name
     * @param classes The method signature classes.
     * @return The method.
     * @throws NoSuchMethodException If the method cannot be located.
     */
    private static Method getMethod(final Class actionClass, final String methodName, Class ...classes)
        throws NoSuchMethodException
    {
        if (methodName != null)
        {
            return actionClass.getMethod(methodName, classes) ;
        }
        else
        {
            return null ;
        }
    }

    /**
     * Get an constructor for the action class.
     * @param actionClass The action class.
     * @return The constructor of the action class.
     * @throws ConfigurationException For errors during instantiation.
     */
    static Constructor getActionClassConstructor(final Class actionClass)
        throws ConfigurationException
    {
        try
        {
            return actionClass.getConstructor(new Class[] {ConfigTree.class}) ;
        }
        catch (final NoSuchMethodException nsme)
        {
            throw new ConfigurationException("Action " + actionClass.getName() + " does not have correct constructor") ;
        }
    }

    /**
     * Get an instance of the action class.
     * @param actionConfig The action class configuration.
     * @param actionClass The action class.
     * @return The instance of the action class.
     * @throws ConfigurationException For errors during instantiation.
     */
    static Object getActionClassInstance(final ConfigTree actionConfig, final Class actionClass)
        throws ConfigurationException
    {
        return getActionClassInstance(actionConfig, getActionClassConstructor(actionClass)) ;
    }

    /**
     * Get an instance of the action class.
     * @param actionConfig The action class configuration.
     * @param actionConstructor The action class constructor.
     * @return The instance of the action class.
     * @throws ConfigurationException For errors during instantiation.
     */
    static Object getActionClassInstance(final ConfigTree actionConfig, final Constructor actionConstructor)
        throws ConfigurationException
    {
        try
        {
            return actionConstructor.newInstance(actionConfig) ;
        }
        catch (final Exception ex)
        {
            throw new ConfigurationException("Unexpected exception while instantiating action instance", ex) ;
        }
    }
}
