/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.message;

import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.helpers.ConfigTree;

/**
 * An abstract {@link MessageComposer} implementation, containing a useful
 * default implementation.
 * <p/>
 * Implementations should be threadsafe (stateless) and must contain
 * a public default constructor.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class AbstractMessageComposer<T> implements MessageComposer<T> {

    private ConfigTree configuration;
    private MessagePayloadProxy payloadProxy;

    /**
     * Set the composers configuration.
     * @param config
     */
    public void setConfiguration(ConfigTree config) {
        this.configuration = config;
        payloadProxy = new MessagePayloadProxy(config,
                new String[] {ActionUtils.POST_ACTION_DATA, Body.DEFAULT_LOCATION, BytesBody.BYTES_LOCATION},
                new String[] {Body.DEFAULT_LOCATION});
    }

    /**
     * Get the composer configuration.
     * @return The composer configuration.
     */
    public ConfigTree getConfiguration() {
        return configuration;
    }

    /**
     * Compose the message.
     * <p/>
     * Override to implement alternative {@link Message} construction strategy.
     *
     * @param messagePayload Message payload to be packaged, or a channel specific
     *                       container class for the message payload (e.g. a JMS message).
     * @return ESB aware message instance.
     * @throws MessageDeliverException Failed to compose message payload for delivery.
     */
    public Message compose(T messagePayload) throws MessageDeliverException {
        Message message = MessageFactory.getInstance().getMessage();

        populateMessage(message, messagePayload);

        return message;
    }

    protected MessagePayloadProxy getPayloadProxy() {
        return payloadProxy;
    }

    /**
     * Decompose the message.
     * <p/>
     * This implementation simple calls Body.get(ActionUtils.POST_ACTION_DATA)}.
     * Override to implement an alternative {@link Message} decomposition strategy.
     *
     * @param message The message to be decomposed.
     * @param originalInputMessagePayload The original input message payload used to
     * compose this (or ther original) message.  The original message can sometimes contain
     * information relevant during the decomposition process.  Whether or not this parameter
     * can be null depends on the MessageComposer implementation.
     * @return The message "task object".
     * @throws MessageDeliverException
     */
    public Object decompose(Message message, T originalInputMessagePayload) throws MessageDeliverException {
        return getPayloadProxy().getPayload(message);
    }

    /**
     * Populate
     *
     * @param message The message instance to be populated.
     * @param messagePayload The message payload to to be populated into the message.
     * @throws MessageDeliverException Unable to populate message with payload.
     */
    protected abstract void populateMessage(Message message, T messagePayload) throws MessageDeliverException;
}
