/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.lifecycle;

import java.util.Collection;

import org.jboss.soa.esb.schedule.ScheduleProvider;
import org.jboss.soa.esb.schedule.SchedulingException;
import org.apache.log4j.Logger;

/**
 * Controller class to manage the lifecycles of a set of managed instances.
 *
 * @author kevin
 */
public class ManagedLifecycleController
{
    private static Logger logger = Logger.getLogger(ManagedLifecycleController.class);   

   /**
    * The managed lifecycle instances.
    */
   private final ManagedLifecycle[] instances;
    private ScheduleProvider scheduleProvider;

    /**
    * Construct the controller to manage the specified managed lifecycle instances.
    *
    * @param lifecycles The managed lifecycle instances.
    */
   public ManagedLifecycleController(final Collection<ManagedLifecycle> lifecycles)
   {
      if(lifecycles == null) {
          instances = new ManagedLifecycle[0];
      } else {
          instances = lifecycles.toArray(new ManagedLifecycle[lifecycles.size()]);
      }
   }

   /**
    * Start the lifecyles for the managed instances.
    * <p/>
    * The managed instances will be left in the destroyed state if an error occurs.
    *
    * @throws ManagedLifecycleException For errors during the lifecycle start.
    */
   public void start()
           throws ManagedLifecycleException
   {
      initialiseInstances();
      startInstances();
      try {
         if(scheduleProvider != null) {
             scheduleProvider.start();
         }
      } catch (SchedulingException e) {
          stopAndDestroy(false);
          throw new ManagedLifecycleException("Failed to start schedule provider.", e);
      }
   }

    /**
    * Restart the lifecyles for the managed instances.
    * <p/>
    * The managed instances will be left in the destroyed state if an error occurs.
    *
    * @throws ManagedLifecycleException For errors during the lifecycle restart.
    */
   public void restart() throws ManagedLifecycleException
   {
      try
      {
          if(scheduleProvider != null) {
              scheduleProvider.standby();
          }
      } catch (SchedulingException e) {
          stopAndDestroy(false);
          throw new ManagedLifecycleException("Failed to put schedule provider into standby.", e);
      }
      stopInstances();
      startInstances();
      try {
          if(scheduleProvider != null) {
              scheduleProvider.start();
         }
      } catch (SchedulingException e) {
          stopAndDestroy(false);
          throw new ManagedLifecycleException("Failed to restart schedule provider.", e);
      }
   }

   /**
    * Stop the lifecyles for the managed instances.
    * <p/>
    * The managed instances will be left in the destroyed state if an error occurs.
    *
    * @throws ManagedLifecycleException For errors during the lifecycle stop.
    */
   public void stop() throws ManagedLifecycleException
   {
      try
      {
          stopAndDestroy(true) ;
      }
      finally
      {
          if(scheduleProvider != null) {
              try {
                  scheduleProvider.stop();
              } catch (SchedulingException e) {
                  logger.error("Failed to stop scheduling.", e);
              }
          }
      }
   }

   /**
    * Initialise the managed instances.
    *
    * @throws ManagedLifecycleException For errors during initialisation.
    */
   private void initialiseInstances()
           throws ManagedLifecycleException
   {
      final int numInstances = instances.length;
      for (int count = 0; count < numInstances; count++)
      {
         final ManagedLifecycle instance = instances[count];
         try
         {
            instance.initialise();
         }
         catch (final ManagedLifecycleException mle)
         {
            if (count > 0)
            {
               destroyInstances(count - 1);
            }
            throw mle;
         }
      }
   }

   /**
    * Start the managed instances.
    *
    * @throws ManagedLifecycleException For errors during starting.
    */
   private void startInstances()
           throws ManagedLifecycleException
   {
      final int numInstances = instances.length;
      for (int count = 0; count < numInstances; count++)
      {
         final ManagedLifecycle instance = instances[count];
         try
         {
            instance.start();
         }
         catch (final ManagedLifecycleException mle)
         {
            if (count > 0)
            {
               stopInstances(count - 1);
               destroyInstances(numInstances - 1);
            }
            throw mle;
         }
      }
   }

   /**
    * Stop the managed instances.
    *
    * @throws ManagedLifecycleException For errors during stopping.
    */
   private void stopInstances()
           throws ManagedLifecycleException
   {
      final int numInstances = instances.length;
      for (int count = numInstances - 1; count >= 0; count--)
      {
         final ManagedLifecycle instance = instances[count];
         try
         {
            instance.stop();
         }
         catch (final ManagedLifecycleException mle)
         {
            if (count > 0)
            {
               stopInstances(count - 1);
               destroyInstances(numInstances - 1);
            }
            throw mle;
         }
      }
   }

   /**
    * Destroy the managed instances.
    *
    * @throws ManagedLifecycleException For errors during destruction.
    */
   private void destroyInstances()
           throws ManagedLifecycleException
   {
      final int numInstances = instances.length;
      for (int count = numInstances - 1; count >= 0; count--)
      {
         final ManagedLifecycle instance = instances[count];
         try
         {
            instance.destroy();
         }
         catch (final ManagedLifecycleException mle)
         {
            if (count > 0)
            {
               destroyInstances(count - 1);
            }
            throw mle;
         }
      }
   }

   /**
    * Silently stop a partial set of the managed instances.
    *
    * @param firstInstance The index of the first instance to stop
    */
   private void stopInstances(final int firstInstance)
   {
      for (int count = firstInstance; count >= 0; count--)
      {
         final ManagedLifecycle instance = instances[count];
         try
         {
            instance.stop();
         }
         catch (final ManagedLifecycleException mle)
         {
         } // Ignore exception
      }
   }

   /**
    * Silently destroy a partial set of the managed instances.
    *
    * @param firstInstance The index of the first instance to stop
    */
   private void destroyInstances(final int firstInstance)
   {
      for (int count = firstInstance; count >= 0; count--)
      {
         final ManagedLifecycle instance = instances[count];
         try
         {
            instance.destroy();
         }
         catch (final ManagedLifecycleException mle)
         {
         } // Ignore exception
      }
   }
   
   /**
    * Stop and destroy all lifecycles, cleaning up resources.
    * @param throwExceptions true if exceptions can be throw, false otherwise.
    */
   private void stopAndDestroy(final boolean throwExceptions)
       throws ManagedLifecycleException
   {
       if (throwExceptions)
       {
           stopInstances() ;
           destroyInstances() ;
       }
       else
       {
           final int numInstances = instances.length;
           stopInstances(numInstances-1) ;
           destroyInstances(numInstances-1) ;
       }
   }

    public void setScheduleProvider(ScheduleProvider scheduleProvider) {
        this.scheduleProvider = scheduleProvider;
    }
}
