/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.jca;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;

import org.jboss.soa.esb.Configurable;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.lifecycle.AbstractManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;

/**
 * @author <a href="bill@jboss.com">Bill Burke</a>
 * @author <a href="kevin.conner@jboss.com">Kevin Conner</a>
 */
abstract class BaseJcaInflow<T> extends AbstractManagedLifecycle
{
   private HashMap<String, String> activationSpec = new HashMap<String, String>();
   private ClassLoader classLoader;
   private Class<T> inflowInterface;
   private Class<T> beanClass;
   private Class<?> messagingType;
   private ActivationBridge bridge;
   private String adapter;
   private boolean isTransacted = true;
   protected T bean;

   @SuppressWarnings("unchecked")
   public BaseJcaInflow(ConfigTree config, Class<T> inflowInterface) throws ConfigurationException
   {
      super(config);

      this.inflowInterface = inflowInterface ;
      ConfigTree spec = config.getFirstChild(JcaConstants.ELEMENT_ACTIVATION_CONFIG);
      for (ConfigTree configProperty : spec.getChildren(JcaConstants.ELEMENT_PROPERTY))
      {
         String name = configProperty.getAttribute(JcaConstants.ATTRIBUTE_NAME);
         String value = configProperty.getAttribute(JcaConstants.ATTRIBUTE_VALUE);
         activationSpec.put(name, value);
      }

      this.classLoader = Thread.currentThread().getContextClassLoader();
      adapter = config.getAttribute(JcaConstants.ATTRIBUTE_ADAPTER);

      String bc = config.getAttribute(JcaConstants.ATTRIBUTE_ENDPOINT_CLASS);
      if (bc == null) throw new ConfigurationException("endpointClass attribute is required for Jca Gateway");
      final Class<?> beanClass ;
      try
      {
         beanClass = classLoader.loadClass(bc) ;
      }
      catch (ClassNotFoundException e)
      {
         throw new ConfigurationException("Unable to load Jca Gateway endpointClass: " + bc, e);
      }
      if (!inflowInterface.isAssignableFrom(beanClass))
          throw new ConfigurationException("endpointClass must implement " + inflowInterface.getSimpleName() + " interface");
     
      this.beanClass = (Class<T>)beanClass ;
      
      String mt = config.getAttribute(JcaConstants.ATTRIBUTE_MESSAGING_TYPE);
      if (mt != null)
      {
         try
         {
            messagingType = classLoader.loadClass(mt);
         }
         catch (ClassNotFoundException e)
         {
            throw new ConfigurationException("Unable to load Jca Gateway messagingType: " + mt, e);
         }
      }
      else
      {
         messagingType = findMessagingType(beanClass);
      }

      String brdg = config.getAttribute(JcaConstants.ATTRIBUTE_JCA_BRIDGE);
      if (brdg != null)
      {
         try
         {
            bridge = (ActivationBridge) classLoader.loadClass(mt).newInstance();
         }
         catch (Exception e)
         {
            throw new ConfigurationException("Unable to load jcaBridge: " + brdg, e);
         }
      }
      else
      {
         bridge = new JBoss42ActivationBridge();
      }
      String transacted = config.getAttribute(JcaConstants.ATTRIBUTE_TRANSACTED);
      if (transacted != null) isTransacted = Boolean.parseBoolean(transacted.trim());
   }

   private Class<?> findMessagingType(Class<?> clazz)
           throws ConfigurationException
   {
      if (clazz.equals(Object.class)) return null;

      Class<?>[] interfaces = clazz.getInterfaces();
      Class<?> type = null;
      for (Class<?> intf : interfaces)
      {
         if (intf.equals(inflowInterface) || intf.equals(Configurable.class)) continue;
         if (type != null)
         {
            throw new ConfigurationException("Unable to guess messagingType interface from endpointClass as the base class implements too many interfaces, specify this explicity");
         }
         type = intf;
      }
      if (type == null) return findMessagingType(clazz.getSuperclass());
      return type;
   }

   protected abstract String getDescription() ;

   protected void doInitialise() throws ManagedLifecycleException
   {
      try
      {
         Constructor<T> beanClassConstructor = beanClass.getConstructor(ConfigTree.class);
         try
         {
            bean = beanClassConstructor.newInstance(getConfig());
         }
         catch (InstantiationException e)
         {
            throw new RuntimeException("Unable to create endpoint bean", e);
         }
         catch (IllegalAccessException e)
         {
            throw new RuntimeException("Unable to create endpoint bean", e);
         }
         catch (InvocationTargetException e)
         {
            throw new RuntimeException("Unable to create endpoint bean", e.getTargetException());
         }
      }
      catch (NoSuchMethodException e)
      {
         try
         {
            bean = beanClass.newInstance();
         }
         catch (Exception e1)
         {
            throw new RuntimeException("Unable to create endpoint bean", e1);
         }
      }
      
      if (bean instanceof Configurable)
      {
          try
          {
              ((Configurable)bean).setConfiguration(getConfig()) ;
          }
          catch (final ConfigurationException ce)
          {
              throw new RuntimeException("Unable to configure endpoint bean", ce);
          }
      }
      
      final Object theBean = bean;
      
      final String description = getDescription() ;
      
      EndpointContainer container = new EndpointContainer()
      {

         public String getDescription()
         {
            return description + " jca adapter: " + adapter;
         }

         public Object invoke(Method method, Object[] args) throws Throwable
         {
            try
            {
               return method.invoke(theBean, args);
            }
            catch (IllegalAccessException e)
            {
               throw new RuntimeException(e);
            }
            catch (InvocationTargetException e)
            {
               throw e.getTargetException();
            }
         }

         public boolean isDeliveryTransacted(Method method) throws NoSuchMethodException
         {
            return isTransacted;
         }
      };


      EndpointFactory mef = new EndpointFactory();
      mef.setContainer(container);
      mef.setLoader(classLoader);
      mef.setMessagingType(messagingType);
      mef.start();
      bridge.setActivationProperties(activationSpec);
      bridge.setAdapter(adapter);
      bridge.setMessageEndpointFactory(mef);
      bridge.setMessagingTypeClass(messagingType);
   }

   protected void doStart()
   {
      bridge.activate();
   }

   protected void doStop() throws ManagedLifecycleException
   {
      bridge.deactivate();
   }
   
   protected void doDestroy() throws ManagedLifecycleException
   {
   }
   
   protected boolean isTransactional()
   {
       return isTransacted ;
   }
}
