/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway;

import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.util.FileUtil;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageComposer;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.internal.soa.esb.assertion.AssertArgument;

import java.io.File;
import java.io.IOException;

/**
 * Local File Message Composer.
 *
 * @see org.jboss.soa.esb.listeners.gateway.RemoteFileMessageComposer
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class LocalFileMessageComposer<T extends File> implements MessageComposer<T> {

    public static final String PROP_FILE_OBJ = "in-file-obj";
    public static final String PROP_FILE_PATH = "in-file-path";
    public static final String PROP_FILE_LENGTH = "in-file-length";
    public static final String PROP_FILE_LASTMOD = "in-file-lastmod";

    private MessagePayloadProxy payloadProxy;

    public void setConfiguration(ConfigTree config) throws ConfigurationException {
        payloadProxy = new MessagePayloadProxy(config,
                new String[] {BytesBody.BYTES_LOCATION},
                new String[] {BytesBody.BYTES_LOCATION});
    }

    public Message compose(T inputFile) throws MessageDeliverException {
        AssertArgument.isNotNull(inputFile, "inputFile");

        if(!inputFile.exists()) {
            throw new MessageDeliverException("Invalid File payload.  File '" + inputFile.getAbsolutePath() + "' doesn't exist.");
        }

        Message message = MessageFactory.getInstance().getMessage();
        try {
            payloadProxy.setPayload(message, FileUtil.readFile(inputFile));
        } catch (IOException e) {
            throw new MessageDeliverException("Error reading input file '" + inputFile.getAbsolutePath() + "'.", e);
        }

        // Add some metadata about the file....
        message.getProperties().setProperty(PROP_FILE_OBJ, inputFile);
        message.getProperties().setProperty(PROP_FILE_PATH, inputFile.getAbsolutePath());
        message.getProperties().setProperty(PROP_FILE_LENGTH, inputFile.length());
        message.getProperties().setProperty(PROP_FILE_LASTMOD, inputFile.lastModified());

        return message;
    }

    public Object decompose(Message message, T inputFile) throws MessageDeliverException {
        return payloadProxy.getPayload(message);
    }
}
