/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway;

import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.*;

import javax.management.MBeanServer;

import org.apache.log4j.Logger;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.marshal.MarshalFactory;
import org.jboss.remoting.marshal.http.HTTPMarshaller;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.lifecycle.AbstractManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.message.AbstractMessageComposer;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter;
import org.jboss.soa.esb.message.MessagePayloadProxy.NullPayloadHandling;
import org.jboss.soa.esb.message.*;
import org.jboss.soa.esb.message.Properties;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.RegistryFactory;
import org.jboss.internal.soa.esb.remoting.HttpUnmarshaller;
import org.jboss.internal.soa.esb.remoting.HttpMarshaller;

/**
 * JBoss Remoting listener implementation for receiving ESB unaware messages
 * over a JBoss Remoting channel.
 * <p/>
 * This class implements the JBoss Remoting interface {@link org.jboss.remoting.ServerInvocationHandler}.
 * Messages are pushed in through the {@link #invoke(org.jboss.remoting.InvocationRequest)} method,
 * which is an implementation of the {@link org.jboss.remoting.ServerInvocationHandler#invoke(org.jboss.remoting.InvocationRequest)}
 * method.
 * <p/>
 * The JBoss Remoting {@link org.jboss.remoting.transport.Connector}
 * configuration is populated by the {@link #initaliseJBRConnectorConfiguration(java.util.Map)}
 * method.  The remoting server locator URI is constructed by the
 * {@link #getJbrServerLocatorURI()} method.
 * <p/>
 * The default {@link org.jboss.soa.esb.listeners.message.MessageComposer} used by this listener is the
 * {@link org.jboss.soa.esb.listeners.gateway.JBossRemotingGatewayListener.JBossRemotingMessageComposer}.  All
 * message composer implementation supplied to this class will be supplied an instance of
 * {@link org.jboss.remoting.InvocationRequest}, from which they must compose an ESB aware
 * {@link Message} instance.
 * <p/>
 * See the <a href="http://labs.jboss.com/portal/jbossremoting">JBoss Remoting</a> docs
 * for details on configuring specific remoting server protocols.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JBossRemotingGatewayListener extends AbstractManagedLifecycle implements ServerInvocationHandler {

    private static final long serialVersionUID = 1L;
    /**
     * JBoss Remoting connector config attribute name prefix.
     */
    public static final String JBR_PREFIX = "jbr-";
    /**
     * Server Protocol config attribute name.
     */
    public static final String JBR_SERVER_PROTOCOL = JBR_PREFIX + "serverProtocol";
    /**
     * Server Host config attribute name.
     */
    public static final String JBR_SERVER_HOST = JBR_PREFIX + ServerInvoker.SERVER_BIND_ADDRESS_KEY;
    /**
     * Server port config attribute name.
     */
    public static final String JBR_SERVER_PORT = JBR_PREFIX + ServerInvoker.SERVER_BIND_PORT_KEY;

    /**
     * Class Logger instance.
     */
    private static Logger logger = Logger.getLogger(JBossRemotingGatewayListener.class);
    /**
     * JBoss Remoting connector.
     */
    private Connector connector;
    /**
     * Connector configuration.
     */
    private Map<String, String> connectorConfig = new HashMap<String, String>();
    /**
     * Server URI.
     */
    private String jbrServerLocatorURI;
    /**
     * Delivery adapter.
     */
    private UncomposedMessageDeliveryAdapter messageDeliveryAdapter;
    /**
     * Simple flag marking the listener instance as being initialised or un-initialised.
     */
    private boolean initialised;
    /**
     * Service category to which this listener is associated.
     */
    private String serviceCategory;
    /**
     * Service name to which this listener is associated.
     */
    private String serviceName;
    /**
     * Listener endpoint EPR.
     */
    private EPR endpointReference;
    /**
     * Is the listener synchronous.
     */
    private boolean synchronous = true;

    /**
     * Install our own marshaller/unmarshaller for HTTP.
     */
    static {
        MarshalFactory.addMarshaller(HTTPMarshaller.DATATYPE,
                new HttpMarshaller(),
                new HttpUnmarshaller());
    }

    /**
     * Construct the threaded managed lifecycle.
     *
     * @param config The configuration associated with this instance.
     * @throws org.jboss.soa.esb.ConfigurationException
     *          for configuration errors during initialisation.
     */
    public JBossRemotingGatewayListener(ConfigTree config) throws ConfigurationException {
        super(config);
        serviceCategory = config.getAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG);
        serviceName = config.getAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG);
        synchronous = !config.getAttribute("synchronous", "true").equalsIgnoreCase("false");
    }

    /**
     * Is this listener instance initialised.
     *
     * @return True if this listener is initialised, otherwise false.
     */
    public boolean isInitialised() {
        return initialised;
    }

    /**
     * Is this listener instance started.
     * <p/>
     * Basically, Is this listener's JBoss Remoting Connector connected?
     *
     * @return True if this listener is started, otherwise false.
     */
    public boolean isStarted() {
        return (connector != null);
    }

    /*
     * ***************************************************************************
     *
     * AbstractManagedLifecycle methods...
     * 
     * ****************************************************************************
     */

    protected void doInitialise() throws ManagedLifecycleException {
        if (isInitialised()) {
            throw new ManagedLifecycleException("Unexpected request to initialise JBoss Remoting Gateway listener '" + getConfig().getName() + "'.  Gateway already initialised.");
        }

        try {
            endpointReference = new EPR(getJbrServerLocatorURI());
            messageDeliveryAdapter = createDeliveryAdapter();
            initaliseJBRConnectorConfiguration(connectorConfig);
        } catch (ConfigurationException e) {
            throw new ManagedLifecycleException("Remoting Listener configuration failed.", e);
        }

        initialised = true;
    }

    protected void doStart() throws ManagedLifecycleException {
        if (!isInitialised()) {
            throw new ManagedLifecycleException("Unexpected request to start JBoss Remoting Gateway listener '" + getConfig().getName() + "'.  Gateway not initialised.");
        }
        if (isStarted()) {
            throw new ManagedLifecycleException("Unexpected request to start JBoss Remoting Gateway listener '" + getConfig().getName() + "'.  Gateway already started.");
        }

        // Start the JBR Server...
        startJBRServer();

        // Regsiter the JBR Endpoint...
        try {
            registerEndpoint();
        } catch (Throwable t) {
            logger.error("Unable to register service endpoint '" + endpointReference.getAddr().getAddress()
                    + "' for service '" + serviceCategory + ":" + serviceName + "'.  Stopping JBossRemoting Server...", t);
            stopJBRServer();
        }
    }

    protected void doStop() throws ManagedLifecycleException {
        if (!isStarted()) {
            throw new ManagedLifecycleException("Unexpected request to stop JBoss Remoting Gateway listener '" + getConfig().getName() + "'.  Gateway not running.");
        }

        unregisterEndpoint();
        stopJBRServer();
    }

    private void startJBRServer() throws ManagedLifecycleException {
        try {
            InvokerLocator locator = new InvokerLocator(jbrServerLocatorURI);

            connector = new Connector(locator, connectorConfig);
            connector.create();
            connector.addInvocationHandler(getConfig().getAttribute("name", this.toString()), this);
            connector.start();

            logger.info("JBoss Remoting Gateway listener '" + getConfig().getName() + "' started.");
        } catch (Throwable throwable) {
            connector = null;
            throw new ManagedLifecycleException("Unable to start Remoting Listener instsance " + getClass().getName(), throwable);
        }
    }

    private void stopJBRServer() throws ManagedLifecycleException {
        try {
            connector.stop();
            logger.info("JBoss Remoting Gateway listener '" + getConfig().getName() + "' stopped.");
        } catch (Throwable throwable) {
            throw new ManagedLifecycleException("Unable to stop Remoting Listener instsance " + getClass().getName(), throwable);
        } finally {
            connector = null;
        }
    }

    private void registerEndpoint() throws ConfigurationException, RegistryException {
        String serviceDescription = getConfig().getAttribute(ListenerTagNames.SERVICE_DESCRIPTION_TAG);
        RegistryFactory.getRegistry().registerEPR(serviceCategory, serviceName, serviceDescription,
                endpointReference, endpointReference.getAddr().getAddress());
    }

    private void unregisterEndpoint() {
        try {
            RegistryFactory.getRegistry().unRegisterEPR(serviceCategory, serviceName, endpointReference);
        } catch (Throwable t) {
            logger.error("Unable to unregister service endpoint '" + endpointReference.getAddr().getAddress()
                    + "' for service '" + serviceCategory + ":" + serviceName + "'.", t);
        }
    }

    protected void doDestroy() throws ManagedLifecycleException {
    }

    /*
     * ***************************************************************************
     *
     * JBoss Remoting ServerInvocationHandler methods...
     *
     * ****************************************************************************
     */

    /**
     * Process a remoting invocation message.
     * <p/>
     * This method uses an {@link org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter}
     * to carry out the delivery.  This delivery adpter is constructed with a
     * {@link org.jboss.soa.esb.listeners.message.MessageComposer} instance supplied through
     * configuration, otherwise it uses the
     * {@link org.jboss.soa.esb.listeners.gateway.JBossRemotingGatewayListener.JBossRemotingMessageComposer}.
     * <p/>
     * The message composer is responsible for mapping the remoting {@link org.jboss.remoting.InvocationRequest}
     * into an ESB aware {@link org.jboss.soa.esb.message.Message}, while the
     * {@link org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter} is responsible for its
     * delivery to the target service.
     *
     * @param invocationRequest JBoss Remoting request.
     * @return Message delivery acknowledgment response.
     * @throws Throwable Message processing failure.
     */
    public Object invoke(InvocationRequest invocationRequest) throws Throwable {
        try {
            if (synchronous) {
                Object response = messageDeliveryAdapter.deliverSync(invocationRequest, 20000); // TODO Fix magic number
                if(logger.isDebugEnabled()) {
                    logger.debug("Returning response [" + response + "].");
                }

                /*
                if(response instanceof String) {
                    response = ((String)response).getBytes("UTF-8");
                }
                */
                
                return response;
            } else {
                messageDeliveryAdapter.deliverAsync(invocationRequest);
            }
        } catch (Throwable t) {
            logger.error("JBoss Remoting Gateway failed to " + (synchronous ? "synchronously" : "asynchronously") + " deliver message to target service [" +
                    messageDeliveryAdapter.getDeliveryAdapter().getServiceCategory() + ":" +
                    messageDeliveryAdapter.getDeliveryAdapter().getServiceName() + "].", t);

            throw t;
        }

        return "<ack/>";
    }

    public void setMBeanServer(MBeanServer mBeanServer) {
    }

    public void setInvoker(ServerInvoker serverInvoker) {
    }

    public void addListener(InvokerCallbackHandler invokerCallbackHandler) {
    }

    public void removeListener(InvokerCallbackHandler invokerCallbackHandler) {
    }

    /**
     * Initialise the JBossRemoting connector configuration.
     * <p/>
     * Constructs the JBR {@link org.jboss.remoting.InvokerLocator} URI
     * through a call to {@link #getJbrServerLocatorURI()}. Also
     * populates the server connector properties.
     * <p/>
     * Default behavior for population of the connector configuration is to load
     * all listener configuration properties whose name is prefixed with "jbr-",
     * stripping off the "jbr-" prefix from the name before adding.
     * So, to set the Server "timeout" configuration property
     * on the connector, you set the property name to "jbr-timeout".
     *
     * @param connectorConfig The configuration map instance to be populated.
     * @throws ConfigurationException Problem populating the configuration.
     */
    protected void initaliseJBRConnectorConfiguration(Map<String, String> connectorConfig) throws ConfigurationException {
        // Initialse the JBR connector URI...
        jbrServerLocatorURI = getJbrServerLocatorURI().toString();

        // Populate the connector config...
        List<KeyValuePair> attributes = getConfig().attributesAsList();
        for (KeyValuePair attribute : attributes) {
            String attributeName = attribute.getKey();

            if (attributeName.startsWith(JBR_PREFIX)) {
                connectorConfig.put(attributeName.substring(JBR_PREFIX.length()), attribute.getValue());
            }
        }
    }

    /**
     * Get the Service Locator URI for this remotng based listener.
     * <p/>
     * Uses the listener config to extract the {@link #JBR_SERVER_PROTOCOL protcol},
     * {@link #JBR_SERVER_HOST host} and {@link #JBR_SERVER_PORT port}
     * parameters for the server.  The host address defaults to
     * the value returned by {@link java.net.InetAddress#getLocalHost()}.
     *
     * @return The Server Locator URI.
     * @throws ConfigurationException One or more of the locator properties
     *                                are missing from the listener config.
     */
    protected URI getJbrServerLocatorURI() throws ConfigurationException {
        String protocol = getConfig().getAttribute(JBR_SERVER_PROTOCOL);
        String host = getConfig().getAttribute(JBR_SERVER_HOST);
        String port = getConfig().getAttribute(JBR_SERVER_PORT);

        if (protocol == null || protocol.trim().equals("")) {
            throw new ConfigurationException("Invalid JBoss Remoting Gateway configuration [" + getConfig().getName() + "]. 'protocol' configuration attribute not specified.");
        }
        if (host == null || host.trim().equals("")) {
            try {
                host = InetAddress.getLocalHost().getHostName();
            } catch (UnknownHostException e) {
                throw new ConfigurationException("Invalid JBoss Remoting Gateway configuration [" + getConfig().getName() + "]. 'host' configuration attribute not specified and unablee to determine local hostname.", e);
            }
        }
        if (port == null || port.trim().equals("")) {
            throw new ConfigurationException("Invalid JBoss Remoting Gateway configuration [" + getConfig().getName() + "]. 'port' configuration attribute not specified.");
        }

        String uriString = protocol + "://" + host + ":" + port;
        try {
            return new URI(uriString);
        } catch (URISyntaxException e) {
            throw new ConfigurationException("Invalid JBoss Remoting Gateway configuration [" + getConfig().getName() + "]. Invalid server locator URI '" + uriString + "'.");
        }
    }

    /**
     * Factory method for adapter creation.
     *
     * @return The adapter instance.
     * @throws ConfigurationException Configuration error.
     */
    protected UncomposedMessageDeliveryAdapter createDeliveryAdapter() throws ConfigurationException {
        return UncomposedMessageDeliveryAdapter.getGatewayDeliveryAdapter(getConfig(), new JBossRemotingMessageComposer<InvocationRequest>());
    }

    /**
     * Message composer for a JBoss Remoting {@link org.jboss.remoting.InvocationRequest}
     * instance.
     */
    public static class JBossRemotingMessageComposer<T extends InvocationRequest> extends AbstractMessageComposer<T> {

        private MessagePayloadProxy payloadProxy;
        
        public void setConfiguration(ConfigTree config) {
            super.setConfiguration(config);
            payloadProxy = new MessagePayloadProxy(config,
                    new String[] {ActionUtils.POST_ACTION_DATA, Body.DEFAULT_LOCATION, BytesBody.BYTES_LOCATION},
                    new String[] {ActionUtils.POST_ACTION_DATA});
            // Allow null to be set on as the message payload...
            payloadProxy.setNullSetPayloadHandling(NullPayloadHandling.LOG);
        }

        protected MessagePayloadProxy getPayloadProxy() {
            return payloadProxy;
        }

        @SuppressWarnings("unchecked")
        protected void populateMessage(Message message, T invocationRequest) throws MessageDeliverException {

            // Set the payload from the JBR invocation...
            payloadProxy.setPayload(message, invocationRequest.getParameter());

            // Copy the request properties onto the message...
            Map properties = invocationRequest.getRequestPayload();
            if (properties != null) {
                // Purposely not iterating over the Map.Entry Set because there's
                // a bug in the Map impl used by JBossRemoting.  Not all the
                // "values" are actually in the Map.Entry set.  Some of them are handled
                // from within an overridden impl of the Map.get(Object) method.
                Set names = properties.keySet();
                for (Object name : names) {
                    Object value = properties.get(name);
                    if(value != null) {
                        message.getProperties().setProperty(name.toString(), value);
                    }
                }
            }
        }

        public Object decompose(Message message, T invocationRequest) throws MessageDeliverException {
            Properties properties = message.getProperties();
            String propertyNames[] = properties.getNames();
            Map responseMap = invocationRequest.getReturnPayload();

            if(responseMap == null) {
                responseMap = new LinkedHashMap();
                invocationRequest.setReturnPayload(responseMap);
            }

            for(String name : propertyNames) {
                Object value = properties.getProperty(name);

                if(value instanceof ResponseHeader) {
                    ResponseHeader header = (ResponseHeader) value;
                    responseMap.put(header.getName(), header.getValue());
                }
            }
            
            return super.decompose(message, invocationRequest);
        }
    }

}
