/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Enumeration;

import javax.jms.ConnectionFactory;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Queue;
import javax.jms.Topic;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.message.Properties;
import org.jboss.soa.esb.notification.jms.JMSPropertiesSetter;

/**
 * Default impl of ESBPropertiesSetter.
 * </p>
 * The following JMS Header fields are set on the ESB Message instance:
 * <lu>
 * <li> JMSMessageID 		-> Call.setMessagID
 * <li> JMSCorrelationID	-> Call.setRelatesTo	URI = jms:correlationID#YYXX-123456780-GG
 * <li> JMSReplyTo			-> Call.setReplyTo
 * <li> JMSExpiration		-> Properties( key = JMSPropertiesSetter.JMS_EXPIRATION )
 * <li> JMSRedeliverd		-> Properties( key = JMSPropertiesSetter.JMS_REDELIVERED
 * </lu>
 * Note: JMSCorrelationID is set as a fragment on the URI and can be retrieved like this:
 * correlationURI.getFragment();
 * 
 * @author <a href="mailto:daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 * @since 4.2
 *
 */
public class DefaultESBPropertiesSetter implements ESBPropertiesSetter
{
	private Logger log = Logger.getLogger( DefaultESBPropertiesSetter.class );

	public void setPropertiesFromJMSMessage( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		setMessageID( fromJMSMessage, toESBMessage );
		setCorrelationID( fromJMSMessage, toESBMessage );
		setReplyTo( fromJMSMessage, toESBMessage );
		setExpiration( fromJMSMessage, toESBMessage );
		setProperties( fromJMSMessage, toESBMessage );
	}
	
	private void setMessageID( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		if ( fromJMSMessage.getJMSMessageID()!=null) 
		{
			try
			{
				toESBMessage.getHeader().getCall().setMessageID( new URI( fromJMSMessage.getJMSMessageID() ));
			} 
			catch (URISyntaxException e) 
			{
				String errorMessage = "Could not set ESB MessageID to JMSMessageID [" + fromJMSMessage.getJMSMessageID() + "]";
				log.error( errorMessage, e );
				throw new JMSException( errorMessage );
			}
		}
	}
	
	private void setCorrelationID( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		if ( fromJMSMessage.getJMSCorrelationID()!=null) 
		{
			try
			{
				final URI correlationURI = new URI( JMSEpr.JMS_PROTOCOL, "correlationID" , fromJMSMessage.getJMSCorrelationID()  );
				toESBMessage.getHeader().getCall().setRelatesTo( correlationURI );
			} 
			catch (URISyntaxException e) 
			{
				final String errorMessage = "Could not ESB setRelatesTo to JMSCorrelationID [ " + fromJMSMessage.getJMSCorrelationID() + "]";
				log.error( errorMessage, e );
				throw new JMSException( errorMessage );
			}
		}
	}
	
	private void setReplyTo( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		if ( fromJMSMessage.getJMSReplyTo()!=null) 
		{
			final Destination replyToDestination = fromJMSMessage.getJMSReplyTo();
			final String connectionFactory = ModulePropertyManager.getPropertyManager("javax.jms.ConnectionFactory").getProperty(ConnectionFactory.class.getName(), "ConnectionFactory");
			String destType = null;
			String destName = null;
			if ( replyToDestination instanceof Queue )
			{
				final Queue queue = ( Queue ) replyToDestination;
				destName = queue.getQueueName();
				destType = JMSEpr.QUEUE_TYPE;
			}
			else
			{
				final Topic topic = ( Topic ) replyToDestination;
				destName = topic.getTopicName();
				destType = JMSEpr.TOPIC_TYPE;
			}
			toESBMessage.getHeader().getCall().setReplyTo( new JMSEpr( destType , destName, connectionFactory ) );
		}
	}
	
	private void setExpiration( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		if( fromJMSMessage.getJMSExpiration() != 0 )
		{
			long ttl = fromJMSMessage.getJMSExpiration() - System.currentTimeMillis();
			if ( ttl < 0 )
				ttl = javax.jms.Message.DEFAULT_TIME_TO_LIVE;
			log.debug( "Setting JMS Expiration : " + ttl );
			toESBMessage.getProperties().setProperty( JMSPropertiesSetter.JMS_EXPIRATION, ttl );
		}
	}
	
	private void setProperties( final Message fromJMSMessage, final org.jboss.soa.esb.message.Message toESBMessage ) throws JMSException
	{
		@SuppressWarnings("unchecked")
		Enumeration<String> properties = fromJMSMessage.getPropertyNames();
		Properties esbProperties = toESBMessage.getProperties();
		if (null != properties)
		{
			while (properties.hasMoreElements())
			{
				String key = properties.nextElement();
				Object value = fromJMSMessage.getObjectProperty(key);
				if (null != value)
					esbProperties.setProperty(key, value);
			}
	     }
		final boolean redelivered = fromJMSMessage.getJMSRedelivered();
		esbProperties.setProperty( JMSPropertiesSetter.JMS_REDELIVERED, Boolean.valueOf( redelivered ));
	}

}
