/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.http;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.http.HttpClientFactory;
import org.apache.commons.httpclient.HttpClient;

import java.util.Properties;
import java.net.URI;
import java.net.URISyntaxException;

/**
 * HttpClient Configurator.
 * <p/>
 * Configure some aspect of a HttpClient instance.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class Configurator {

    public abstract void configure(HttpClient httpClient, Properties properties) throws ConfigurationException;

    /**
     * Gets a {@link URI} for the {@link HttpClientFactory#TARGET_HOST_URL} property,
     * if set.
     * @param properties Properties.
     * @param required If the {@link HttpClientFactory#TARGET_HOST_URL} property is required.  An
     * exception is thrown if this is true and the property is not present.
     * @return URI for the {@link HttpClientFactory#TARGET_HOST_URL} property, otherwise null
     * if not set.
     * @throws ConfigurationException {@link HttpClientFactory#TARGET_HOST_URL} property is not set
     * an required is true, or the configured URI is invalid.
     */
    protected URI getTargetURI(Properties properties, boolean required) throws ConfigurationException {
        String targetURLString = properties.getProperty(HttpClientFactory.TARGET_HOST_URL);

        if(required) {
            assertPropertySetAndNotBlank(targetURLString, HttpClientFactory.TARGET_HOST_URL);
        }
        if(targetURLString != null) {
            try {
                return new URI(targetURLString);
            } catch (URISyntaxException e) {
                throw new ConfigurationException("'url' property is not a valid URI.");
            }
        }

        return null;
    }

    /**
     * Assert that the property value is not null and not blank.
     * @param propertyValue The value of the property.
     * @param propertyName  The name of the property.
     */
    protected void assertPropertySetAndNotBlank(String propertyValue, String propertyName) throws ConfigurationException {
        if(propertyValue == null || propertyValue.trim().equals("")) {
            throw new ConfigurationException(getClass().getSimpleName() + " must be configured with a non-blank value for the '" + propertyName + "' property.");
        }
    }

    /**
     * Assert that the property value is an Integer.
     * @param propertyValue The value of the property.
     * @param propertyName  The name of the property.
     */
    protected void assertPropertyIsInteger(String propertyValue, String propertyName) throws ConfigurationException {
        assertPropertySetAndNotBlank(propertyValue, propertyName);
        try {
            Integer.parseInt(propertyValue);
        } catch(NumberFormatException e) {
            throw new ConfigurationException(getClass().getSimpleName() + " must be configured with an Integer value for the '" + propertyName + "' property.");
        }
    }
}
