/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.helpers;

import java.io.Serializable;
import java.util.List;

public class KeyValuePair implements Serializable {
	private static final long serialVersionUID = 1L;

	private String mKey, mVal;

	public KeyValuePair(String pKey, String pVal) {
		if(pKey == null || pKey.trim().equals("")) {
			throw new IllegalArgumentException("null or empty 'pKey' arg in call.");
		}
		
		mKey = pKey;
		mVal = pVal;
	}

	public String getKey() {
		return mKey;
	}

	public String getValue() {
		return mVal;
	}

	public String toString() {
		return mVal;
	}

	public String dump() {
		return "KVpair[" + mKey + "=" + mVal + "]";
	}
    
    /**
     * Get the value associated with the specified key from the supplied list of Key Value Pairs.
     * <p/>
     * Returns the value from the first matching key.
     * @param key The key to search for.
     * @param list The list of KeyValuePairs to search.
     * @return The value associated with the supplied key, or null if key not found.
     */
    public static String getValue(String key, List<KeyValuePair> list) {
        if(key == null) {
            throw new IllegalArgumentException("null 'key' arg in call.");
        }
        if(list == null) {
            throw new IllegalArgumentException("null 'list' arg in call.");
        }
        
        for(KeyValuePair kvp : list) {
            if(kvp.mKey.equals(key)) {
                return kvp.mVal;
            }
        }
        
        return null;
    }
    
    /**
     * Get the value associated with the specified key from the supplied list of Key Value Pairs.
     * <p/>
     * Returns the value from the first matching key.
     * @param key The key to search for.
     * @param list The list of KeyValuePairs to search.
     * @param defaultVal The default value to be returned where there's no value available for the specified key. 
     * @return The value associated with the supplied key, or null if key not found.
     */
    public static String getValue(String key, List<KeyValuePair> list, String defaultVal) {
        String value = getValue(key, list);
        
        if(value == null) {
            return defaultVal;
        }
        
        return value;
    }
    
    /**
     * Get the boolean equivalent value associated with the specified key 
     * from the supplied list of Key Value Pairs.
     * <p/>
     * Returns:
     * <ul>
     *  <li><b><code>true</code></b>: If value equals "true" or "yes" or "y" (ignoring case).</li>
     *  <li><b><code>false</code></b>: If value equals "false" or "no" or "n" (ignoring case).</li>
     *  <li><b><i>defaultVal</i></b>: If none of the above hold true.</li>
     * </ul>
     * <p/>
     * Returns the value from the first matching key.
     * @param key The key to search for.
     * @param list The list of KeyValuePairs to search.
     * @param defaultVal The default value to be returned where the above listed conditions do not hold
     * for the associated value, or the value is not specified.
     * @return The boolean equivalent value associated with the specified key according to the above specified
     * rules, otherwise the <b><i>defaultVal</i></b> is returned.
     */
    public static boolean getBooleanValue(String key, List<KeyValuePair> list, boolean defaultVal) {
        String value = getValue(key, list);
        
        if(value == null) {
            return defaultVal;
        }
        
        if(value.equalsIgnoreCase("true") || value.equalsIgnoreCase("y") || value.equalsIgnoreCase("yes")) {
            return true;
        } else if(value.equalsIgnoreCase("false") || value.equalsIgnoreCase("n") || value.equalsIgnoreCase("no")) {
            return false;
        } else {
            return defaultVal;
        }
    }

    /**
     * Get the numeric <code>double</code> equivalent value associated with the specified key 
     * from the supplied list of Key Value Pairs.
     * <p/>
     * Returns the value from the first matching key.
     * @param key The key to search for.
     * @param list The list of KeyValuePairs to search.
     * @param defaultVal The default value to be returned where the value is not found or is non-numeric.
     * @return The <code>double</code> equivalent value associated with the specified key if the value is found
     * and is numeric, otherwise the <b><i>defaultVal</i></b> is returned.
     */
    public static double getDoubleValue(String key, List<KeyValuePair> list, double defaultVal) {
        String value = getValue(key, list);
        
        if(value == null) {
            return defaultVal;
        }

        try {
            return Double.parseDouble(value);
        } catch(NumberFormatException e) {
            // return the default...
        }
        
        return defaultVal;
    }
}
