/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.common;

import javax.transaction.Synchronization;
import javax.transaction.xa.XAResource;

/**
 * This class represents the transaction strategy that is currently in force
 * within the ESB.  At present there are two strategies employed, a null strategy
 * when running outside of an application server environment and a JTA strategy
 * when running within.
 * 
 * @author <a href='mailto:kevin.conner@jboss.com'>Kevin Conner</a>
 */
public abstract class TransactionStrategy
{
    /**
     * The null transaction strategy.
     */
    private static final TransactionStrategy NULL_TRANSACTION_STRATEGY = new NullTransactionStrategy() ;
    /**
     * The current transaction strategy
     */
    private static volatile TransactionStrategy transactionStrategy = NULL_TRANSACTION_STRATEGY ;
    
    /**
     * Get the active transaction strategy.
     * @param transacted True if the current transacted strategy is required, false for a null strategy.
     * @return The transaction strategy.
     */
    public static TransactionStrategy getTransactionStrategy(final boolean transacted)
    {
        return (transacted ? transactionStrategy : NULL_TRANSACTION_STRATEGY) ;
    }
    
    /**
     * Set the active transaction strategy.
     * @param transactionStrategy The transaction strategy.
     */
    public static void setTransactionStrategy(final TransactionStrategy transactionStrategy)
    {
        if (transactionStrategy == null)
        {
            TransactionStrategy.transactionStrategy = NULL_TRANSACTION_STRATEGY ;
        }
        else
        {
            TransactionStrategy.transactionStrategy = transactionStrategy ;
        }
    }
    
    /**
     * Begin a transaction on the current thread.
     * @throws TransactionStrategyException
     */
    public abstract void begin()
        throws TransactionStrategyException ; 
    
    /**
     * Terminate the transaction on the current thread.
     * If the transaction has been marked for rollback then it
     * will be rolled back, otherwise it will be committed.
     * @throws TransactionStrategyException
     */
    public abstract void terminate()
        throws TransactionStrategyException ; 
    
    /**
     * Mark the current transaction for rollback.
     * @throws TransactionStrategyException
     */
    public abstract void rollbackOnly()
        throws TransactionStrategyException ;
    
    /**
     * Get a handle on the currently associated transaction (or null).
     * @throws TransactionStrategyException
     */
    
    public abstract Object getTransaction () throws TransactionStrategyException;
    
    /**
     * Suspend the current thread-to-transaction association.
     * 
     * @return the associated transaction, or null.
     * @throws TransactionStrategyException
     */
    public abstract Object suspend () throws TransactionStrategyException;
    
    /**
     * Is the currently associated transaction active?
     * @return
     * @throws TransactionStrategyException
     */
    public abstract boolean isActive () throws TransactionStrategyException;
    
    /**
     * Associated the transaction with the current thread.
     * @param tx
     * @throws TransactionStrategyException
     */
    public abstract void resume (Object tx) throws TransactionStrategyException;
    
    /**
     * Add a synchronization to the current transaction.
     * @param sync
     * @throws TransactionStrategyException
     */
    public abstract void registerSynchronization (Synchronization sync) throws TransactionStrategyException;
    
    /**
     * Add a resource to the current transaction.
     * @param resource
     * @throws TransactionStrategyException
     */
    public abstract void enlistResource(XAResource resource) throws TransactionStrategyException;
    
    /**
     * The null transaction strategy.
     * @author kevin
     */
    private static class NullTransactionStrategy extends TransactionStrategy
    {
        /**
         * Begin a transaction on the current thread.
         * @throws TransactionStrategyException
         */
        public void begin()
            throws TransactionStrategyException
        {
        }
        
        /**
         * Terminate the transaction on the current thread.
         * If the transaction has been marked for rollback then it
         * will be rolled back, otherwise it will be committed.
         * @throws TransactionStrategyException
         */
        public void terminate()
            throws TransactionStrategyException
        {
        }
        
        /**
         * Mark the current transaction for rollback.
         * @throws TransactionStrategyException
         */
        public void rollbackOnly()
            throws TransactionStrategyException
        {
        }
        
        /**
         * Get a handle on the currently associated transaction (or null).
         * @throws TransactionStrategyException
         */
        
        public Object getTransaction () throws TransactionStrategyException
        {
        	return null;
        }
        
        /**
         * Suspend the current thread-to-transaction association.
         * 
         * @return the associated transaction, or null.
         * @throws TransactionStrategyException
         */
        public Object suspend () throws TransactionStrategyException
        {
        	return null;
        }
        
        /**
         * Associated the transaction with the current thread.
         * @param tx
         * @throws TransactionStrategyException
         */
        public void resume (Object tx) throws TransactionStrategyException
        {
            throw new TransactionStrategyException("Unsupported in this transaction strategy") ;
        }
        
        /**
         * Add a synchronization to the current transaction.
         * @param sync
         * @throws TransactionStrategyException
         */
        public void registerSynchronization (Synchronization sync) throws TransactionStrategyException
        {
            throw new TransactionStrategyException("Unsupported in this transaction strategy") ;
        }
        
        /**
         * Add a resource to the current transaction.
         * @param resource
         * @throws TransactionStrategyException
         */
        public void enlistResource(XAResource resource) throws TransactionStrategyException
        {
            throw new TransactionStrategyException("Unsupported in this transaction strategy") ;
        }
        
        /**
         * Is the currently associated transaction active?
         * @return
         * @throws TransactionStrategyException
         */
        public boolean isActive () throws TransactionStrategyException
        {
        	return false;
        }
        
        public String toString ()
        {
        	return "NullTransactionStrategy";
        }
    }
}