package org.jboss.soa.esb.addressing.util;

import java.net.MalformedURLException;
import java.net.URISyntaxException;

import org.jboss.internal.soa.esb.addressing.eprs.DefaultFileReplyToEpr;
import org.jboss.internal.soa.esb.addressing.eprs.DefaultFtpReplyToEpr;
import org.jboss.internal.soa.esb.addressing.eprs.DefaultHibernateReplyToEpr;
import org.jboss.internal.soa.esb.addressing.eprs.DefaultJdbcReplyToEpr;
import org.jboss.internal.soa.esb.addressing.eprs.DefaultJmsReplyToEpr;
import org.jboss.internal.soa.esb.addressing.eprs.DefaultSftpReplyToEpr;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.addressing.eprs.HibernateEpr;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.addressing.eprs.SFTPEpr;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.message.Message;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * Provide some helper routines for common requirements.
 */

public class DefaultReplyTo
{
    /**
         * Obtain a default replyToEPR (based on the toEpr) to which we can
         * reply to.
         * 
         * @param toEpr -
         *                the toEPR which will be used a basis to generate the
         *                replyToEPR.
         * @return EPR - the EPR to which the pickup will be delivered.
         * @throws URISyntaxException,
         *                 CourierException
         */
    
    public static EPR getReplyTo (EPR toEpr) throws CourierException,
	    MalformedEPRException
    {
	try
	{
	    if (null == toEpr)
		throw new IllegalArgumentException("toEpr must not be null");
	    if (toEpr instanceof JMSEpr)
		return new DefaultJmsReplyToEpr((JMSEpr) toEpr);
	    // Check for SFTPEpr first, as it extends FTPEpr
	    if (toEpr instanceof SFTPEpr)
		return new DefaultSftpReplyToEpr((SFTPEpr) toEpr);
	    // Check for FTPEpr first, as it extends FileEpr
	    if (toEpr instanceof FTPEpr)
		return new DefaultFtpReplyToEpr((FTPEpr) toEpr);
	    if (toEpr instanceof FileEpr)
		return new DefaultFileReplyToEpr((FileEpr) toEpr);
	    if (toEpr instanceof JDBCEpr)
		return new DefaultJdbcReplyToEpr((JDBCEpr) toEpr);
	    if (toEpr instanceof HibernateEpr)
		return new DefaultHibernateReplyToEpr((HibernateEpr) toEpr);
	}
	catch (MalformedURLException ex)
	{
	    throw new MalformedEPRException(ex);
	}
	catch (URISyntaxException ex)
	{
	    throw new MalformedEPRException(ex);
	}

	throw new CourierException("Unsupported default reply to EPR "
		+ toEpr.getClass().getSimpleName());
    }

    /**
     * Initialise the message as a reply to the specified call details.
     * @param message The reply message.
     * @param callDetails The call details used to generate the reply.
     * @return true if the message was initialised, false otherwise.
     */
	public static boolean initialiseReply(final Message message, final Call callDetails)
	{
		return initialiseReply(message, callDetails, getReplyTo(callDetails)) ;
	}
	
    /**
     * Get the appropriate replyTo address..
     * @param callDetails The call details used to generate the reply.
     * @return The replyTo address or null if none set.
     */
	static EPR getReplyTo(final Call callDetails)
	{
		if (callDetails == null)
		{
			return null ;
		}
		
		final EPR replyTo = callDetails.getReplyTo() ;
		if (replyTo != null)
		{
			return replyTo ;
		}
		else
		{
			return callDetails.getFrom() ;
		}
	}
	
    /**
     * Initialise the message as a reply to the specified call details.
     * @param message The reply message.
     * @param callDetails The call details used to generate the reply.
     * @param toEPR The target EPR for the message.
     * @return true if the message was initialised, false otherwise.
     */
	static boolean initialiseReply(final Message message, final Call callDetails,
		final EPR toEPR)
	{
		if ((message == null) || (callDetails == null) || (toEPR == null))
		{
			return false ;
		}
		
		final Call replyCallDetails = message.getHeader().getCall() ;
		replyCallDetails.setTo(toEPR) ;
		replyCallDetails.setFrom(callDetails.getTo()) ;
		replyCallDetails.setReplyTo(null) ;
		replyCallDetails.setFaultTo(null) ;
		replyCallDetails.setRelatesTo(callDetails.getMessageID()) ;
		replyCallDetails.setMessageID(null) ;
		
		return true ;
	}
}