package org.jboss.soa.esb.addressing.eprs;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */


/**
 * This class represents the endpoint reference for services.
 */

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.addressing.XMLUtil;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * A helper class for using file based  EPRs. Simply create instances of this
 * class instead of the base EPR. Since URLs can use file:// as the protocol, we try to leverage
 * that as much as possible.
 * 
 * @author marklittle
 *
 */
public class FileEpr extends EPR
{
	public static final String INPUT_SUFFIX_TAG = "inputSuffix";
	public static final String WORK_SUFFIX_TAG 	= "workSuffix";
	public static final String POST_DIR_TAG 	= "postDir";
	public static final String POST_SUFFIX_TAG 	= "postSuffix";
	public static final String POST_DEL_TAG 	= "postDelete";
	public static final String ERROR_DIR_TAG 	= "errorDir";
	public static final String ERROR_SUFFIX_TAG	= "errorSuffix";
	public static final String ERROR_DEL_TAG 	= "errorDelete";
	public static final String URL_TAG = "URL";
	
	public static final String DEFAULT_REPLY_TO_FILE_SUFFIX = "_reply";
	
	public FileEpr (EPR epr)
	{
		super(epr);
		
		if (epr instanceof FileEpr)
		{
		    FileEpr instance = (FileEpr) epr;
		    
		    inputSet = instance.inputSet;
		    workSet = instance.workSet;
		    postDirSet = instance.postDirSet;
		    postSuffixSet = instance.postSuffixSet;
		    postDelSet = instance.postDelSet;
		    errorDirSet = instance.errorDirSet;
		    errorSuffixSet = instance.errorSuffixSet;
		    errorDelSet = instance.errorDelSet;
		}
	}
	
	public FileEpr (EPR epr, Element header)
	{
		super(epr);
		
		NodeList nl = header.getChildNodes();

		for (int i = 0; i < nl.getLength(); i++)
		{
			try
			{
				String prefix = nl.item(i).getPrefix();
				String tag = nl.item(i).getLocalName();
				
				if ((prefix != null) && (prefix.equals(XMLUtil.JBOSSESB_PREFIX)))
				{
					if (tag != null)
					{
						if (tag.equals(INPUT_SUFFIX_TAG))
						    setInputSuffix(nl.item(i).getTextContent());
						else
						{
							if (tag.equals(POST_DIR_TAG))
							    setPostDirectory(nl.item(i).getTextContent());
							else
							{
								if (tag.equals(POST_SUFFIX_TAG))
								    setPostSuffix(nl.item(i).getTextContent());
								else
								{
									if (tag.equals(POST_DEL_TAG))
									{
									    String content = nl.item(i).getTextContent();
									    
									    if ("true".equalsIgnoreCase(content))
										setPostDelete(true);
									    else
									    {
										if ("false".equalsIgnoreCase(content))
										    setPostDelete(false);
									    }
									}
									else
									{
										if (tag.equals(ERROR_DIR_TAG))
										{
										    setErrorDirectory(nl.item(i).getTextContent());
										}
										else
										{
											if (tag.equals((ERROR_SUFFIX_TAG)))
											    setErrorSuffix(nl.item(i).getTextContent());
											else
											{
												if (tag.equals((ERROR_DEL_TAG)))
												{
												    String content = nl.item(i).getTextContent();
												    
												    if ("true".equalsIgnoreCase(content))
													setErrorDelete(true);
												    else
												    {
													if ("false".equalsIgnoreCase(content))
													    setErrorDelete(false);
												    }											
												}
											}
										}
									}
								}
							}
						}	
					}
				}
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}

	public FileEpr (URL url) throws URISyntaxException
	{
		super(new URI(url.toString()));
	}
	
	public FileEpr (String url) throws URISyntaxException
	{
		super(new URI(url));
	}

	/**
	 * Set the URL for this endpoint.
	 * 
	 * @param url the address.
	 */
	
	public void setURL (URL url)
	{
		super.setAddr(new PortReference(url.toString()));
	}
	
	/**
	 * Get the URL address.
	 * 
	 * @return the address.
	 * @throws URISyntaxException thrown if the address is invalid.
	 */
	
	public URL getURL () throws MalformedURLException, URISyntaxException
	{
		return new URL(super.getAddr().getAddress());
	}

	/**
	 * Set the file input suffix.
	 * 
	 * @param suffix the input suffix to use.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setInputSuffix (String suffix) throws URISyntaxException
	{
		if (suffix == null)
			throw new IllegalArgumentException();
		
		if (inputSet)
			throw new IllegalStateException("Input suffix already set.");
		
		getAddr().addExtension(INPUT_SUFFIX_TAG, suffix);
		inputSet = true;
	}
	
	/**
	 * @return the input suffix associated with this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getInputSuffix () throws URISyntaxException
	{
		return getAddr().getExtensionValue(INPUT_SUFFIX_TAG);
	}
	
	/**
	 * Set the work suffix for this EPR.
	 * 
	 * @param suffix the suffix to use.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setWorkSuffix (String suffix) throws URISyntaxException
	{
		if (suffix == null)
			throw new IllegalArgumentException();
		
		if (workSet)
			throw new IllegalStateException("Cannot change work suffix");
		
		getAddr().addExtension(WORK_SUFFIX_TAG, suffix);
		workSet = true;
	}
	
	/**
	 * @return the work suffix associated with this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getWorkSuffix () throws URISyntaxException
	{
		return getAddr().getExtensionValue(WORK_SUFFIX_TAG);
	}
	
	/**
	 * Set the post directory for this EPR.
	 * 
	 * @param dir the directory to use.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setPostDirectory (String dir) throws URISyntaxException
	{
		if (dir == null)
			throw new IllegalArgumentException();
		
		if (postDirSet)
			throw new IllegalStateException("Cannot change post directory");
		
		getAddr().addExtension(POST_DIR_TAG, dir);
		postDirSet = true;
	}
	
	/**
	 * @return the post directory associated with this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getPostDirectory () throws URISyntaxException
	{
		return getAddr().getExtensionValue(POST_DIR_TAG);
	}
	
	/**
	 * Set the post suffix for this EPR.
	 * 
	 * @param suffix the suffix to use.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setPostSuffix (String suffix) throws URISyntaxException
	{
		if (suffix == null)
			throw new IllegalArgumentException();
		
		if (postSuffixSet)
			throw new IllegalStateException("Cannot change post suffix");
		
		getAddr().addExtension(POST_SUFFIX_TAG, suffix);
		postSuffixSet = true;
	}
	
	/**
	 * @return the post suffix associated with this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getPostSuffix () throws URISyntaxException
	{
		return getAddr().getExtensionValue(POST_SUFFIX_TAG);
	}
	
	/**
	 * Set the post delete for this EPR.
	 * 
	 * @param del the deleted value to use.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setPostDelete (boolean del) throws URISyntaxException
	{		
		if (postDelSet)
			throw new IllegalStateException("Cannot change post delete");
		
		if (del)
			getAddr().addExtension(POST_DEL_TAG, "true");
		else
			getAddr().addExtension(POST_DEL_TAG, "false");
		
		postDelSet = true;
	}
	
	/**
	 * @return the delete vazlue associated with this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final boolean getPostDelete () throws URISyntaxException
	{
		return ("true".equals(getAddr().getExtensionValue(POST_DEL_TAG)));
	}
		
	/**
	 * Set the error directory for this EPR.
	 * 
	 * @param dir the directory to use.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setErrorDirectory (String dir) throws URISyntaxException
	{
		if (dir == null)
			throw new IllegalArgumentException();
		
		if (errorDirSet)
			throw new IllegalStateException("Cannot change error directory");
		
		getAddr().addExtension(ERROR_DIR_TAG, dir);
		errorDirSet = true;
	}
	
	/**
	 * @return the error directory associated with this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getErrorDirectory () throws URISyntaxException
	{
		return getAddr().getExtensionValue(ERROR_DIR_TAG);
	}
	
	/**
	 * Set the error suffix for this EPR.
	 * 
	 * @param suffix the suffix to use.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setErrorSuffix (String suffix) throws URISyntaxException
	{
		if (suffix == null)
			throw new IllegalArgumentException();
		
		if (errorSuffixSet)
			throw new IllegalStateException("Cannot change error suffix");
		
		getAddr().addExtension(ERROR_SUFFIX_TAG, suffix);
		errorSuffixSet = true;
	}
	
	/**
	 * @return the error suffix associated with this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getErrorSuffix () throws URISyntaxException
	{
		return getAddr().getExtensionValue(ERROR_SUFFIX_TAG);
	}
	
	/**
	 * Set the error delete for this EPR. (invalid Message files will be deleted)
	 * 
	 * @param del the deleted value to use.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setErrorDelete (boolean del) throws URISyntaxException
	{		
		if (errorDelSet)
			throw new IllegalStateException("Cannot change error delete");
		
		if (del)
			getAddr().addExtension(ERROR_DEL_TAG, "true");
		else
			getAddr().addExtension(ERROR_DEL_TAG, "false");
		
		errorDelSet = true;
	}
	
	/**
	 * Attention - Default 
	 * @return the error delete value associated with this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final boolean getErrorDelete () throws URISyntaxException
	{
		return (! "false".equals(getAddr().getExtensionValue(ERROR_DEL_TAG)));
	}
		
	public String toString ()
	{
		return "FileEpr [ "+super.getAddr().extendedToString()+" ]";
	}

	public EPR copy ()
	{
	    return new FileEpr(this);
	}
	
	public static URI type ()
	{
	    return _type;
	}
	
	protected FileEpr (URI uri)
	{
		super(uri);
	}

	private boolean inputSet = false;
	private boolean workSet = false;
	private boolean postDirSet = false;
	private boolean postSuffixSet = false;
	private boolean postDelSet = false;
	private boolean errorDirSet = false;
	private boolean errorSuffixSet = false;
	private boolean errorDelSet = false;
	
	private static URI _type;
	
	static
	{
	    try
		{
		    _type = new URI("urn:jboss/esb/epr/type/file");
		}
		catch (Exception ex)
		{
		    ex.printStackTrace();
		    
		    throw new ExceptionInInitializerError(ex.toString());
		}
	}
}