package org.jboss.soa.esb.actions.converters;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathExpressionException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.util.XPathUtil;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.xml.DomDriver;
import com.thoughtworks.xstream.io.xml.DomReader;


/**
 * Object to processor.
 * <p/>
 * Uses the <a href="http://xstream.codehaus.org/">XStream</a> processor to generate an XML message String from the supplied object.
 * <p/>
 * Sample Action Configuration:
 * <pre>
 * &lt;Action name="doCustomer" processor="XStreamObject"&gt;
 *     &lt;property name="class-alias" value="Customer" /&gt; &lt;!-- Optional. Class alias used in call to <a href="http://xstream.codehaus.org/javadoc/com/thoughtworks/xstream/XStream.html">XStream.alias(String, Class)</a> prior to deserialisation. --&gt;
 *     &lt;property name="incoming-type" value="CustomerProcessor" /&gt; &lt;!-- Required. Class for incoming type used to process the message after  deserialisation. --&gt;
 *     &lt;property name="exclude-package" value="false" /&gt; &lt;!-- Optional. Default "true".  Not applicable if a "class-alias" is specified. --&gt;
 *     &lt;property name="root-node" value="/root/Customer" /&gt; 
 *     &lt;!-- Optional. Specify an XPath expression be used to determine the root node used with XStream. 
 *     Useful when the object to convert is not the root node of the document --&gt;
 *     &lt;property name="aliases"&gt; &lt;!-- Optional list of extra aliases to add to XStream  --&gt;
 * 		&lt;alias name="aliasName" class="className" /&gt; 
 * 		&lt;alias name="aliasName" class="className" /&gt; 
 * 		...
 *     &lt;/property&gt;
 * &lt;/Action&gt;
 * </pre>
 * <p/>
 * The XML root element is either set from the "class-alias" property or the classes full name.  In the later case, the class package is
 * excluded unless "exclude-package" is set to "false"/"no". 
 * 
 * This can be used with ObjectToXStream
 * 
 * @author danielmarchant
 * @author Daniel Bevenius
 * @since Version 4.0
 */
public class XStreamToObject  extends AbstractObjectXStream {

	private static Logger logger = Logger.getLogger(XStreamToObject.class);
	
    // class related variables
    private Class incomingType;
    
    // action related variables
	private Map<String,String> aliases;
    private MessagePayloadProxy payloadProxy;

    /**
     * Public constructor.
     * @param properties Action Properties.
     * @throws ConfigurationException Action not properly configured.
     */
    public XStreamToObject(ConfigTree properties) {
    	this(properties.getName(), properties.attributesAsList());
    	aliases = getAliases( properties );
        payloadProxy = new MessagePayloadProxy(properties,
                                               new String[] {BytesBody.BYTES_LOCATION, ActionUtils.POST_ACTION_DATA},
                                               new String[] {ActionUtils.POST_ACTION_DATA});
    }
    
    /**
     * Public constructor.
     * @param actionName Action name.
     * @param properties Action Properties.
     * @throws ConfigurationException Action not properly configured.
     */
    protected XStreamToObject(String actionName, List<KeyValuePair> properties) {
    	super(actionName,properties);
    	String incomingTypeStr = KeyValuePair.getValue("incoming-type", properties);
    	try {
			incomingType = ClassUtil.forName(incomingTypeStr, getClass());
		} catch (ClassNotFoundException e) {
			logger.error("Could not find : " + incomingTypeStr,e);
		}
    }
	
	/**
	 * Processes the message by using the giving class-processor.
	 *  
	 */
	public Message process(Message message) throws ActionProcessingException {
        Object object;

        try {
            object = payloadProxy.getPayload(message);
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }

        try {
			Object toObject = incomingType.newInstance();
			toObject = fromXmlToObject( object.toString(), toObject );
			
			payloadProxy.setPayload(message, toObject);
		} catch (InstantiationException e) {
			logger.error( e );
			throw new ActionProcessingException("Could not invoke for Arg: " + getName(),e );
		} catch (IllegalAccessException e) {
			logger.error( e );
			throw new ActionProcessingException("Could not access for Arg: " + getName(),e );
		} catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }

        return message;
	}
	
	/**
	 * Will extract the alias elements from the passed-in conifgTree 
	 * 
	 * @param configTree			the configuration for this class
	 * 
	 * @return Map<String,String> 	either an empty map or a map containing the alias name
	 * 								as its key and the corresponding value is the class to map
	 * 								it to.	
	 */
	protected Map<String,String> getAliases( ConfigTree configTree )
	{
		Map<String,String> aliases = new HashMap<String,String>();
		
		ConfigTree[] children = configTree.getChildren( "alias" );
		
		if ( children != null ) {
			for ( ConfigTree alias : children )
				aliases.put( alias.getAttribute( "name" ), alias.getAttribute( "class" ) );
		}
		return aliases;
	}

	/**
	 * Added the aliases contained in the passed-in map to the
	 * passed-in XStream object
	 * 
	 * @param aliases	 Map of aliases.
	 * @throws ActionProcessingException 
	 */
	protected void addAliases( Map<String, String> aliases, XStream xstream) throws ActionProcessingException
	{
		if ( aliases == null )
			return;
		
		Set<Map.Entry<String,String>> set = aliases.entrySet();
		for (Map.Entry me : set ) {
			String className = (String) me.getValue();
			try {
				Class clazz = ClassUtil.forName( className, getClass() );
		        xstream.alias((String)me.getKey(), clazz );
			} catch (ClassNotFoundException e) {
				logger.error(e);
				throw new ActionProcessingException("Could not add alias : " + (String)me.getKey() + ", class : " + className ,e );
			}
		}
	}
	
	/**
	 * 
	 * @param xml		the xml String
	 * @param root		an instance of the type of the root element
	 * @throws ActionProcessingException
	 * @throws ParserConfigurationException 
	 * @throws IOException 
	 * @throws SAXException 
	 */
	protected Object fromXmlToObject(String xml, Object root ) throws ActionProcessingException
	{
		HierarchicalStreamReader reader = null;
		try
		{
			XStream xstream = new XStream( new DomDriver() );
			
			reader = new DomReader( getRootElement( xml, rootNodeName ) );
				
	        xstream.alias(getAlias(incomingType), incomingType);
	        addAliases( aliases, xstream );
			return xstream.unmarshal( reader, root );
		}
		finally 
		{
			if ( reader != null)  reader.close();
		}
	}

	/*
	 * Simply delegates to XPathUtil and catches exceptions specific
	 * to that class and rethrows an ActionProcessingException
	 */
	private Element getRootElement( String xml, String xPathExpression ) throws ActionProcessingException
	{
		logger.debug( "rootNodeName : " + xPathExpression );
		
		try
		{
			return XPathUtil.getNodeFromXPathExpression( xml, xPathExpression );
		} 
		catch (ParserConfigurationException e)
		{
			logger.error( e );
			throw new ActionProcessingException( e );
		} 
		catch (SAXException e)
		{
			logger.error( e );
			throw new ActionProcessingException( e );
		} 
		catch (IOException e)
		{
			logger.error( e );
			throw new ActionProcessingException( e );
		}
		catch (XPathExpressionException e)
		{
			logger.error( e );
			throw new ActionProcessingException( e );
		}
	}
	
}
