/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.converters;

import java.util.List;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;

import com.thoughtworks.xstream.XStream;

/**
 * Object to XML processor.
 * <p/>
 * Uses the <a href="http://xstream.codehaus.org/">XStream</a> processor to generate an XML message String from the supplied object.
 * <p/>
 * Sample Action Configuration:
 * <pre>
 * &lt;Action name="Customer-To-XML" processor="ObjectToXStream"&gt;
 *     &lt;property name="class-alias" value="Customer" /&gt; &lt;!-- Class alias used in call to <a href="http://xstream.codehaus.org/javadoc/com/thoughtworks/xstream/XStream.html">XStream.alias(String, Class)</a> prior to serialisation. --&gt;
 *     &lt;property name="exclude-package" value="false" /&gt; &lt;!-- Default "true".  Not applicable if a "class-alias" is specified. --&gt;
 * &lt;/Action&gt;
 * </pre>
 * <p/>
 * The XML root element is either set from the "class-alias" property or the classes full name.  In the later case, the class package is
 * excluded unless "exclude-package" is set to "false"/"no". 
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @since Version 4.0
 */
public class ObjectToXStream extends AbstractActionPipelineProcessor {
    
    private String classAlias;
    private boolean excludePackage;
    private MessagePayloadProxy payloadProxy;

    /**
     * Public constructor.
     * @param configTree Action Properties.
     * @throws ConfigurationException Action not properly configured.
     */
    public ObjectToXStream(ConfigTree configTree) {
        payloadProxy = new MessagePayloadProxy(configTree,
                                               new String[] {BytesBody.BYTES_LOCATION, ActionUtils.POST_ACTION_DATA},
                                               new String[] {ActionUtils.POST_ACTION_DATA});
        classAlias = configTree.getAttribute("class-alias");
        excludePackage = configTree.getAttribute("exclude-package", "true").equals("true");
    }

    /* (non-Javadoc)
     * @see org.jboss.soa.esb.actions.ActionProcessor#process(java.lang.Object)
     */
    public Message process(Message message) throws ActionProcessingException {

        Object object;
        try {
            object = payloadProxy.getPayload(message);
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }

        XStream xstream = new XStream();
        
        if(classAlias == null) {
            if(excludePackage) {
                xstream.alias(object.getClass().getSimpleName(), object.getClass());
            } else {
                xstream.alias(object.getClass().getName(), object.getClass());
            }
        } else {
            xstream.alias(classAlias, object.getClass());
        }
        
        try {
            payloadProxy.setPayload(message, xstream.toXML(object));
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }

        return message;
    }
}
