/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.services.registry;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.RegistryException;
/**
 * Mock class for the Registry.
 * If need be we can extract the interface from here, add a factory and have JAXR as a plugin, allowing
 * for other RegistryAPIs.
 *
 * @author Kurt Stam
 */
public class MockRegistryImpl implements Registry
{
    private static Map<String,Category> categories = new HashMap<String,Category>();
    
	/** 
	 * Publish an EPR to the Registry
	 */
	protected Service registerService(String categoryName, String serviceName, String serviceDescription)
	{
        Category category = categories.get(categoryName);
		if (category==null) {
            category = new Category();
		    categories.put(categoryName, category);
        }
        Service service = new Service();
        service.setName(serviceName);
        service.setDescription(serviceDescription);
        category.getServices().add(service);
		return service;
	}
	/** 
	 * Remove an EPR from the Registry
	 */
	@SuppressWarnings("unchecked")
    public void unRegisterService(String categoryName, String serviceName) throws RegistryException
    {
        Category category = categories.get(categoryName);
        if (category==null) {
            throw new RegistryException("Service not found");
        } else {
            List<Service> services = category.getServices();
            for (Service service : services) {
                if (serviceName.equals(service.getName())) {
                    services.remove(service);
                }
            }
        }
	}
	/** 
	 * Publish an EPR to the Registry
	 */
	public void registerEPR(String categoryName, String serviceName, String serviceDescription, EPR epr, String eprDescription) 
		throws RegistryException
	{
		Service service = registerService(categoryName, serviceName, serviceDescription);
        ServiceBinding serviceBinding = new ServiceBinding();
        serviceBinding.setDescription(eprDescription);
        serviceBinding.setEpr(epr);
        service.getServiceBindings().add(serviceBinding);
    }
	/** 
	 * Remove an EPR from the Registry
	 */
	public void unRegisterEPR(String category, String serviceName, EPR toBeDeletedEPR) throws RegistryException{
		//first find the ServiceBindings for this service
        Service service = findService(category, serviceName);
        List<ServiceBinding> serviceBindings = service.getServiceBindings();
        for (ServiceBinding binding : serviceBindings) {
            if (toBeDeletedEPR.equals(binding.getEpr())) {
                serviceBindings.remove(binding);
            }
        }
	}
	
	/** 
	 * {@inheritDoc}
	 * @return collection services
	 */
	public List<String> findAllServices() throws RegistryException
	{
        List<String> serviceNames = new ArrayList<String>();
        for (Category category : categories.values()) {
            for (Service service : category.getServices()) {
                serviceNames.add(service.name);
            }
        }
        return serviceNames;
	}
	/**
	 * Find Services based on a category ("transformation").
	 * 
	 * @param serviceType
	 * @return List services
	 */
	public List<String> findServices(String categoryName) throws RegistryException
	{
        List<String> serviceNames = new ArrayList<String>();
		Category category = categories.get(categoryName);
        if (category==null) {
            throw new RegistryException("No category called " + categoryName + " exist");
        }
        for (Service service : category.getServices()) {
            serviceNames.add(service.name);
        }
        return serviceNames;
	}
	/**
	 * 
	 * @param service
	 * @return
	 */
	public List<EPR> findEPRs(String categoryName, String serviceName) throws RegistryException
	{
		List<EPR> eprs = new ArrayList<EPR>();
        Service service = findService(categoryName, serviceName);
        for (ServiceBinding serviceBinding : service.serviceBindings) {
            eprs.add(serviceBinding.getEpr());
        }
        return eprs;
	}
	/**
	 * 
	 * @param service
	 * @return
	 */
	public EPR findEPR(String categoryName, String serviceName) throws RegistryException
	{
        Service service = findService(categoryName, serviceName);
        for (ServiceBinding serviceBinding : service.serviceBindings) {
            //return the first one in the list
            return serviceBinding.getEpr();
        }
        return null;
	}
	/**
	 * 
	 * @param category
	 * @param serviceName
	 * @return Service
	 */
	protected synchronized static Service findService(String categoryName, String serviceName) throws RegistryException
	{
        Category category = categories.get(categoryName);
        for (Service service : category.getServices()) {
            if (serviceName.equals(service.getName())) {
                return service;
            }
        }
        throw new RegistryException("No service called " + serviceName + " exists in category " + categoryName);
	}
	/**
	 * Finds all services for a given category.
	 * @param category
	 * @param serviceName
	 * @return Service
	 */
	protected synchronized static Collection<Service> findServicesForCategory(String categoryName)
	{
        Category category = categories.get(categoryName);
        return category.getServices();
	}
    
    class Category
    {
        List<Service> services = new ArrayList<Service>();
        
        public List<Service> getServices() {
            return services;
        }
        public void setServices(List<Service> services) {
            this.services = services;
        }
    }
    
    class Service
    {
        String name;
        String description;
        List<ServiceBinding> serviceBindings = new ArrayList<ServiceBinding>();
        
        public String getDescription() {
            return description;
        }
        public void setDescription(String description) {
            this.description = description;
        }
        public String getName() {
            return name;
        }
        public void setName(String name) {
            this.name = name;
        }
        public List<ServiceBinding> getServiceBindings() {
            return serviceBindings;
        }
        public void setServiceBindings(List<ServiceBinding> serviceBindings) {
            this.serviceBindings = serviceBindings;
        }
    }
    
    class ServiceBinding
    {
        String name;
        String description;
        EPR epr;
        
        public String getDescription() {
            return description;
        }
        public void setDescription(String description) {
            this.description = description;
        }
        public EPR getEpr() {
            return epr;
        }
        public void setEpr(EPR epr) {
            this.epr = epr;
        }
        public String getName() {
            return name;
        }
        public void setName(String name) {
            this.name = name;
        }
    }
	
}

	
