/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.rosetta.pooling;

import java.io.Serializable;
import java.util.HashSet;

import javax.jms.BytesMessage;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MapMessage;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.MessageProducer;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueBrowser;
import javax.jms.Session;
import javax.jms.StreamMessage;
import javax.jms.TemporaryQueue;
import javax.jms.TemporaryTopic;
import javax.jms.TextMessage;
import javax.jms.Topic;
import javax.jms.TopicSubscriber;

/**
 * Wrapper for JMS session class, responsible for tracking resources and the pooling.
 */
public class JmsSession implements Session
{
    /**
     * The session delegate.
     */
    private final Session session ;
    
    /**
     * The set of active queue browsers.
     */
    private HashSet<QueueBrowser> queueBrowserSet ;
    /**
     * The set of active message consumers.
     */
    private HashSet<MessageConsumer> messageConsumerSet ;
    /**
     * The set of active message producers.
     */
    private HashSet<MessageProducer> messageProducerSet ;
    
    /**
     * Create the session wrapper.
     * @param session The session delegate.
     * @param isJTA True if this tales part in a JTA transaction
     * @throws JMSException
     */
    JmsSession(final Session session)
        throws JMSException
    {
        this.session = session ;
    }

    public void close() throws JMSException
    {
        session.close();
    }

    public void commit() throws JMSException
    {
        session.commit();
    }

    public QueueBrowser createBrowser(Queue arg0, String arg1)
            throws JMSException
    {
        return trackQueueBrowser(session.createBrowser(arg0, arg1));
    }

    public QueueBrowser createBrowser(Queue arg0) throws JMSException
    {
        return trackQueueBrowser(session.createBrowser(arg0));
    }

    public BytesMessage createBytesMessage() throws JMSException
    {
        associate() ;
        return session.createBytesMessage();
    }

    public MessageConsumer createConsumer(Destination arg0, String arg1,
            boolean arg2) throws JMSException
    {
        return trackMessageConsumer(session.createConsumer(arg0, arg1, arg2));
    }

    public MessageConsumer createConsumer(Destination arg0, String arg1)
            throws JMSException
    {
        return trackMessageConsumer(session.createConsumer(arg0, arg1));
    }

    public MessageConsumer createConsumer(Destination arg0) throws JMSException
    {
        return trackMessageConsumer(session.createConsumer(arg0));
    }

    public TopicSubscriber createDurableSubscriber(Topic arg0, String arg1,
            String arg2, boolean arg3) throws JMSException
    {
        return trackTopicSubscriber(session.createDurableSubscriber(arg0, arg1, arg2, arg3));
    }

    public TopicSubscriber createDurableSubscriber(Topic arg0, String arg1)
            throws JMSException
    {
        return trackTopicSubscriber(session.createDurableSubscriber(arg0, arg1));
    }

    public MapMessage createMapMessage() throws JMSException
    {
        associate() ;
        return session.createMapMessage();
    }

    public Message createMessage() throws JMSException
    {
        associate() ;
        return session.createMessage();
    }

    public ObjectMessage createObjectMessage() throws JMSException
    {
        associate() ;
        return session.createObjectMessage();
    }

    public ObjectMessage createObjectMessage(Serializable arg0)
            throws JMSException
    {
        associate() ;
        return session.createObjectMessage(arg0);
    }

    public MessageProducer createProducer(Destination arg0) throws JMSException
    {
        return trackMessageProducer(session.createProducer(arg0));
    }

    public Queue createQueue(String arg0) throws JMSException
    {
        associate() ;
        return session.createQueue(arg0);
    }

    public StreamMessage createStreamMessage() throws JMSException
    {
        associate() ;
        return session.createStreamMessage();
    }

    public TemporaryQueue createTemporaryQueue() throws JMSException
    {
        associate() ;
        return session.createTemporaryQueue();
    }

    public TemporaryTopic createTemporaryTopic() throws JMSException
    {
        associate() ;
        return session.createTemporaryTopic();
    }

    public TextMessage createTextMessage() throws JMSException
    {
        associate() ;
        return session.createTextMessage();
    }

    public TextMessage createTextMessage(String arg0) throws JMSException
    {
        associate() ;
        return session.createTextMessage(arg0);
    }

    public Topic createTopic(String arg0) throws JMSException
    {
        associate() ;
        return session.createTopic(arg0);
    }

    public int getAcknowledgeMode() throws JMSException
    {
        associate() ;
        return session.getAcknowledgeMode();
    }

    public MessageListener getMessageListener() throws JMSException
    {
        associate() ;
        return session.getMessageListener();
    }

    public boolean getTransacted() throws JMSException
    {
        associate() ;
        return session.getTransacted();
    }

    public void recover() throws JMSException
    {
        associate() ;
        session.recover();
    }

    public void rollback() throws JMSException
    {
        session.rollback();
    }

    public void run()
    {
        session.run();
    }

    public void setMessageListener(MessageListener arg0) throws JMSException
    {
        associate() ;
        session.setMessageListener(arg0);
    }

    public void unsubscribe(String arg0) throws JMSException
    {
        associate() ;
        session.unsubscribe(arg0);
    }

    private synchronized QueueBrowser trackQueueBrowser(QueueBrowser queueBrowser)
        throws JMSException
    {
        associate() ;
        if (queueBrowserSet == null)
        {
            queueBrowserSet = new HashSet<QueueBrowser>() ;
        }
        final QueueBrowser result = getQueueBrowser(queueBrowser) ;
        queueBrowserSet.add(result) ;
        return result ;
    }

    private synchronized MessageConsumer trackMessageConsumer(MessageConsumer messageConsumer)
        throws JMSException
    {
        associate() ;
        if (messageConsumerSet == null)
        {
            messageConsumerSet = new HashSet<MessageConsumer>() ;
        }
        final MessageConsumer result = getMessageConsumer(messageConsumer) ;
        messageConsumerSet.add(result) ;
        return result ;
    }

    private synchronized TopicSubscriber trackTopicSubscriber(TopicSubscriber topicSubscriber)
        throws JMSException
    {
        associate() ;
        if (messageConsumerSet == null)
        {
            messageConsumerSet = new HashSet<MessageConsumer>() ;
        }
        final TopicSubscriber result = getTopicSubscriber(topicSubscriber) ;
        messageConsumerSet.add(result) ;
        return result ;
    }

    private synchronized MessageProducer trackMessageProducer(MessageProducer messageProducer)
        throws JMSException
    {
        associate() ;
        if (messageProducerSet == null)
        {
            messageProducerSet = new HashSet<MessageProducer>() ;
        }
        final MessageProducer result = getMessageProducer(messageProducer) ;
        messageProducerSet.add(result) ;
        return result ;
    }

    synchronized void releaseResources()
    {
        if (queueBrowserSet != null)
        {
            for(QueueBrowser queueBrowser: queueBrowserSet)
            {
                try {
                    queueBrowser.close() ;
                } catch (final JMSException jmse) {} // ignore
            }
            queueBrowserSet = null ;
        }
        if (messageConsumerSet != null)
        {
            for(MessageConsumer messageConsumer: messageConsumerSet)
            {
                try {
                    messageConsumer.close() ;
                } catch (final JMSException jmse) {} // ignore
            }
            messageConsumerSet = null ;
        }
        if (messageProducerSet != null)
        {
            for(MessageProducer messageProducer: messageProducerSet)
            {
                try {
                    messageProducer.close() ;
                } catch (final JMSException jmse) {} // ignore
            }
            messageProducerSet = null ;
        }
        try
        {
            recover() ;
        }
        catch (final JMSException jmse) {} // ignore
    }

    protected QueueBrowser getQueueBrowser(QueueBrowser queueBrowser)
    {
        return queueBrowser ;
    }

    protected MessageConsumer getMessageConsumer(MessageConsumer messageConsumer)
    {
        return messageConsumer ;
    }

    protected TopicSubscriber getTopicSubscriber(TopicSubscriber topicSubscriber)
    {
        return topicSubscriber ;
    }

    protected MessageProducer getMessageProducer(MessageProducer messageProducer)
    {
        return messageProducer ;
    }

    protected void handleCloseSession(final JmsConnectionPool jmsConnectionPool)
    {
        jmsConnectionPool.handleCloseSession(this) ;
    }

    protected void handleReleaseSession(JmsConnectionPool jmsConnectionPool)
    {
        jmsConnectionPool.handleReleaseSession(this) ;
    }
    
    protected void associate()
        throws JMSException
    {
    }
}
