/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.rosetta.pooling;

import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;

import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.lifecycle.LifecyclePriorities;
import org.jboss.soa.esb.lifecycle.LifecycleResource;
import org.jboss.soa.esb.lifecycle.LifecycleResourceException;
import org.jboss.soa.esb.lifecycle.LifecycleResourceFactory;
import org.apache.log4j.Logger;

/**
 * This is a temporary pooling class until we start using JCA. It is designed to build
 * a JMS session pool for each ConnectionFactory.
 * 
 * @author kstam
 * Date: March 13, 2007
 */
public class JmsConnectionPoolContainer{

    protected static Logger logger = Logger.getLogger(JmsConnectionPoolContainer.class);

    /**
     * The lifecycle resource factory.
     */
    private static final LifecycleResourceFactory<Map<Map<String, String>, JmsConnectionPool>> lifecycleJMSPoolFactory = new JmsConnectionPoolFactory() ;
    /**
     * Lifecycle JMS connection pools.
     */
    private static final LifecycleResource<Map<Map<String, String>, JmsConnectionPool>> lifecycleJMSPoolResource =
        new LifecycleResource<Map<Map<String, String>, JmsConnectionPool>>(lifecycleJMSPoolFactory,
                LifecyclePriorities.JMS_CONNECTION_POOL_PRIORITY) ;

    protected static final HashMap<Object, JmsConnectionPool> transactions = new HashMap<Object, JmsConnectionPool>();
    
    /**
     * Returns the pool given a JMSEpr.
     * 
     * @param jmsEpr
     * @return JmsSessionPool
     * @throws ConnectionException
     * @throws URISyntaxException
     */
    public static JmsConnectionPool getPool(JMSEpr jmsEpr) throws ConnectionException, URISyntaxException
    {
        return getPool(jmsEpr.getJndiEnvironment(), 
        		jmsEpr.getConnectionFactory(), 
                jmsEpr.getJMSSecurityPrincipal(), 
                jmsEpr.getJMSSecurityCredential()) ;
    }
    /**
     * Returns the pool given the identifiers for the JMS provider.
     * 
     * @param enviroment 			- JNDI environment for which a JMSConnectionPool should be retreived
     * @param connectionFactory		- connectionfactory for which a JMSConnectionPool should be retreived
     * @return <code>JmsConnectionPool</code>
     * @throws ConnectionException
     */
    public static JmsConnectionPool getPool(Properties enviroment, String connectionFactory)
        throws ConnectionException
    {
    	return getPool(enviroment, connectionFactory, null, null);
    }
    
    /**
     * Returns the pool given the identifiers for the JMS provider.
     * 
     * @param enviroment 			- JNDI evironment for which a JMSConnectionPool should be retreived
     * @param connectionFactory		- connectionfactory for which a JMSConnectionPool should be retreived
     * @param username				- username that should be used to create the JMS Connection
     * @param password				- password that should be used to create the JMS Connection
     * @return <code>JmsConnectionPool</code>
     * @throws ConnectionException
     */
    public static JmsConnectionPool getPool(Properties enviroment, String connectionFactory,
    		final String username, final String password)
        throws ConnectionException
    {
        Map<String,String> poolKey = createPoolKey(enviroment, connectionFactory, username, password);
        final Map<Map<String, String>, JmsConnectionPool> poolMap = getMap() ;
  
        if (poolMap.containsKey(poolKey)) {
            return poolMap.get(poolKey);
        } else {
            JmsConnectionPool pool = new JmsConnectionPool(poolKey);
            
            poolMap.put(poolKey, pool);
            
            return pool;
        }
    }
    /**
     * Creates a poolKey using the identifying parameters
     * 
     * @param environment - the JNDI environment parameters
     * @param connectionFactory
     * @return
     */
    public static Map<String, String> createPoolKey(Properties  environment, String connectionFactory) 
    {
    	return createPoolKey( environment, connectionFactory, null, null );
    }
    /**
     * Creates a poolKey using the identifying parameters
     * 
     * @param environment - the JNDI environment parameters
     * @param connectionFactory
     * @param username		the JMS username to be used. Used with  {@link javax.jms.ConnectionFactory } createConnection
     * @param password		the JMS password to be used. Used with  {@link javax.jms.ConnectionFactory } createConnection
     * @return
     */
    public static Map<String, String> createPoolKey(Properties  environment, String connectionFactory,
    		final String username, final String password) 
    {
        Map<String,String> poolKey = new HashMap<String,String>();
        if (environment!=null) {
            for (Object key : environment.keySet()) {
                poolKey.put(key.toString(), environment.getProperty(key.toString()));
            }
        }
        
        if ( username != null && (!username.equals("null")) && password != null && (!password.equals("null")) ) {
            poolKey.put(JMSEpr.JMS_SECURITY_PRINCIPAL_TAG, username);
            poolKey.put(JMSEpr.JMS_SECURITY_CREDENTIAL_TAG, password);
        }
        
        if (connectionFactory!=null)  poolKey.put(JMSEpr.CONNECTION_FACTORY_TAG, connectionFactory);
        
        return poolKey;
    }
    /**
     * Returns the JNDI environment in the key.
     * @param poolKey
     * @return
     */
    public static Properties getJndiEnvironment(Map<String,String> poolKey) {
        Properties environment = new Properties();
        for (String key: poolKey.keySet()) {
            if (key.startsWith("java.naming.")) {
                environment.setProperty(key, poolKey.get(key));
            }
        }
        return environment;
    }
    /**
     * Gets the number of pools in the constainer.
     */
    public static int getNumberOfPools()
    {
        try
        {
            return lifecycleJMSPoolResource.getLifecycleResource().size() ;
        }
        catch (final LifecycleResourceException lre)
        {
            // The resource has already been destroyed
            return 0 ;
        }
    }
    /**
     * Removes the poolKey from the poolMap.
     * 
     * @param poolKey
     */
    protected static void removePool(Map<String, String> poolKey) {
        try {
            final Map<Map<String, String>, JmsConnectionPool> poolMap = lifecycleJMSPoolResource.getLifecycleResource() ;
            poolMap.remove(poolKey);
        } catch (final LifecycleResourceException lre) {
            if (logger.isDebugEnabled()) {
                logger.debug("Unexpected lifecycle resource exception removing pool", lre) ;
            }
        }
        
    }
    /**
     * Adds this poolKey and pool to the the poolMap.
     * 
     * @param poolKey
     */
    protected static void addToPool(Map<String, String> poolKey, JmsConnectionPool pool)
        throws ConnectionException
    {
        final Map<Map<String, String>, JmsConnectionPool> poolMap = getMap() ;
        if (!poolMap.containsKey(poolKey)) {
            poolMap.put(poolKey, pool);
        }
    }
    
    private static Map<Map<String, String>, JmsConnectionPool> getMap()
        throws ConnectionException
    {
        try {
            return lifecycleJMSPoolResource.getLifecycleResource() ;
        } catch (final LifecycleResourceException lre) {
            throw new ConnectionException("Unexpected lifecycle resource exception", lre) ;
        }
    }
    
    private static class JmsConnectionPoolFactory implements LifecycleResourceFactory<Map<Map<String, String>, JmsConnectionPool>>
    {
        /**
         * Create a resource object which will be associated with the specified lifecycle identity.
         * @param lifecycleIdentity The associated lifecycle identity.
         * @return The lifecycle resource
         * @throws LifecycleResourceException for errors during construction.
         */
        public Map<Map<String, String>, JmsConnectionPool> createLifecycleResource(final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            return new ConcurrentHashMap<Map<String, String>, JmsConnectionPool>() ;
        }
        
        /**
         * Destroy a resource object which is associated with the specified lifecycle identity.
         * @param resource The lifecycle resource.
         * @param lifecycleIdentity The associated lifecycle identity.
         * @return The lifecycle resource.
         * @throws LifecycleResourceException for errors during destroy.
         */
        public void destroyLifecycleResource(final Map<Map<String, String>, JmsConnectionPool> resource,
            final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            for (JmsConnectionPool pool : resource.values()) {
                pool.removeSessionPool();
            }
        }
    }
}