/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.rosetta.pooling;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Session;
import javax.jms.XAConnection;
import javax.jms.XAConnectionFactory;
import javax.naming.Context;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * Interface that needs to be implemented to provide pool of connections.
 * @see DefaultConnectionPoolImpl
 * Default implementation of Connection Pool
 * @author kstam
 * @author <a href="mailto:daniel.bevenius@gmail.com">Daniel Bevenius</a>				
 * Date: March 10, 2007
 */
public class JmsConnectionPool
{
	private static final int DEFAULT_POOL_SIZE = 20;
	private static final int DEFAULT_SLEEP = 30;
	
	private static int CONFIGURED_POOL_SIZE = DEFAULT_POOL_SIZE;
	private static int CONFIGURED_SLEEP = DEFAULT_SLEEP;
	
    /** Maximum number of Sessions that will be created in this pool */
    private int MAX_SESSIONS = DEFAULT_POOL_SIZE;    //TODO Make this manageable
    
    /** Time to sleep when trying to get a session. */
    private int SLEEP_TIME = DEFAULT_SLEEP;
    
    /** Number of free sessions in the pool that can be given out. Indexed by session key */
    private Map<Integer,ArrayList<JmsSession>> freeSessionsMap = new HashMap<Integer,ArrayList<JmsSession>>();
    
    /** Number of session that are currently in use. Indexed by session key mode */
    private Map<Integer,ArrayList<JmsSession>> inUseSessionsMap = new HashMap<Integer,ArrayList<JmsSession>>();
    
    /** Reference to a Queue or Topic Connection, we only need one per pool */
    protected Connection jmsConnection ;
    
    /** The Indentifier of the pool */
    private Map<String, String> poolKey;

    /**
     * Mapping from transactions to sessions.
     */
    private Map<Object, JmsXASession> transactionsToSessions = new HashMap<Object, JmsXASession>() ;
    /**
     * Mapping from sessions to transactions.
     */
    private Map<JmsXASession, Object> sessionsToTransactions = new HashMap<JmsXASession, Object>() ;

    /** Logger */
    private Logger logger = Logger.getLogger(this.getClass());
    
    /**
     * The flag representing XA aware connections.
     */
    private boolean isXAAware ;
    /**
     * Flag signifying that the pool has been terminated.
     */
    private boolean terminated ;
    
    /**
     * Contructor of the pool.
     * 
     */
    public JmsConnectionPool(Map<String, String> poolKey) 
    {
    	this(poolKey, JmsConnectionPool.CONFIGURED_POOL_SIZE, JmsConnectionPool.CONFIGURED_SLEEP);
    }
    
    public JmsConnectionPool(Map<String, String> poolKey, int poolSize, int sleepTime) 
    {
        this.poolKey = poolKey;
        
        MAX_SESSIONS = poolSize;
        SLEEP_TIME = sleepTime;
        
        freeSessionsMap.put(Session.AUTO_ACKNOWLEDGE, new ArrayList<JmsSession>() );
        freeSessionsMap.put(Session.CLIENT_ACKNOWLEDGE, new ArrayList<JmsSession>() );
        freeSessionsMap.put(Session.DUPS_OK_ACKNOWLEDGE, new ArrayList<JmsSession>() );
        
        inUseSessionsMap.put(Session.AUTO_ACKNOWLEDGE, new ArrayList<JmsSession>() );
        inUseSessionsMap.put(Session.CLIENT_ACKNOWLEDGE, new ArrayList<JmsSession>() );
        inUseSessionsMap.put(Session.DUPS_OK_ACKNOWLEDGE, new ArrayList<JmsSession>() );
    }
   
    /**
     * This is where we create the sessions. 
     * 
     * @param poolKey
     * @param transacted
     * @throws JMSException
     */
    private  synchronized void addAnotherSession(Map<String, String> poolKey, final boolean transacted, final int acknowledgeMode)
        throws JMSException
    {
        //Create a new Session
        ArrayList<JmsSession> freeSessions = freeSessionsMap.get(acknowledgeMode);
        // For now we only support JTA transacted sessions
        final JmsSession session ;
        if (transacted) {
            session = new JmsXASession(this, ((XAConnection)jmsConnection).createXASession());
        } else {
            session = new JmsSession(jmsConnection.createSession(transacted, acknowledgeMode));
        }
        freeSessions.add(session);
        logger.debug("Number of Sessions in the pool with acknowledgeMode: " + acknowledgeMode + " is now " + getSessionsInPool(acknowledgeMode));
    }

    /**
     *  This method can be called whenever a connection is needed from the pool.
     *  
     * @return Connection to be used
     * @throws ConnectionException
     */
    public synchronized JmsSession getSession(final int acknowledgeMode) throws NamingException, JMSException, ConnectionException
    {
        try {
            initConnection() ;
        } catch (final NamingContextException nce) {
            throw new ConnectionException("Unexpected exception accessing Naming Context", nce) ;
        }
        final boolean transacted ;
        try {
            transacted = (isXAAware && TransactionStrategy.getTransactionStrategy(true).isActive()) ;
        } catch (final TransactionStrategyException tse) {
            throw new ConnectionException("Failed to determine current transaction context", tse) ;
        }
        
        if (transacted)
        {
            final JmsXASession currentSession = getXASession() ;
            if (currentSession != null)
            {
                return currentSession ;
            }
        }
        final int mode = (transacted ? Session.SESSION_TRANSACTED : acknowledgeMode) ;
        
        final long end = System.currentTimeMillis() + (SLEEP_TIME * 1000) ;
        boolean emitExpiry = logger.isDebugEnabled() ;
        for(;;) {
            ArrayList<JmsSession> freeSessions = freeSessionsMap.get(mode );
            ArrayList<JmsSession> inUseSessions = inUseSessionsMap.get(mode);
            if (freeSessions.size() > 0)
            {
                final JmsSession session = freeSessions.remove(freeSessions.size()-1);
                inUseSessions.add(session);
                return session ;
            } else if (inUseSessions.size()<MAX_SESSIONS) {
                addAnotherSession(poolKey, transacted, mode);
                continue ;
            } else {
                if (emitExpiry)
                {
                    logger.debug("The connection pool was exhausted, waiting for a session to be released.") ;
                    emitExpiry = false ;
                }
                final long now = System.currentTimeMillis() ;
                final long delay = (end - now) ;
                if (delay <= 0)
                {
                    throw new ConnectionException("Could not obtain a JMS connection from the pool after "+SLEEP_TIME+"s.");
                }
                else
                {
                    try
                    {
                        wait(delay) ;
                    }
                    catch (final InterruptedException ie) {} // ignore
                }
            }
        }
    }
    /**
     *  This method can be called whenever a Session is needed from the pool.
     * @return
     * @throws NamingException
     * @throws JMSException
     * @throws ConnectionException
     */
    public JmsSession getSession() throws NamingException, JMSException, ConnectionException
    {
        return getSession(Session.AUTO_ACKNOWLEDGE);
    }
    
    /**
     * This method closes an open connection and returns the connection to the pool.
     * @param session The connection to be returned to the pool.
     * @throws SQLException
     * @deprecated
     */
    public void closeSession(Session session){
        if (session instanceof JmsSession) {
            closeSession((JmsSession)session) ;
        } else {
            logger.error("Invalid JMS Session type in closeSession: " + session);
        }
    }
    
    /**
     * This method closes an open connection and returns the connection to the pool.
     * @param session The connection to be returned to the pool.
     * @throws SQLException
     */
    public void closeSession(JmsSession session){
        session.handleCloseSession(this) ;
    }
    
    /**
     * Handle the real work of closing the connection.
     * @param session The session to close.
     */
    synchronized void handleCloseSession(final JmsSession session)
    {
        final int mode ;
        try {
            mode = session.getAcknowledgeMode() ;
        } catch (final JMSException jmse) {
            logger.warn("JMSException while calling getAcknowledgeMode") ;
            logger.debug("JMSException while calling getAcknowledgeMode", jmse) ;
            return ;
        }
        
        final ArrayList<JmsSession> sessions = (freeSessionsMap == null ? null : freeSessionsMap.get(mode));
        if (sessions != null) {
            sessions.add(session) ;
        }
        handleReleaseSession(session) ;
    }
    
    /**
     * Handle the real work of releasing the connection.
     * @param session The session to release.
     */
    synchronized void handleReleaseSession(final JmsSession session)
    {
        session.releaseResources() ;
        final int mode ;
        try {
            mode = session.getAcknowledgeMode() ;
        } catch (final JMSException jmse) {
            logger.warn("JMSException while calling getAcknowledgeMode") ;
            logger.debug("JMSException while calling getAcknowledgeMode", jmse) ;
            return ;
        }
        
        final ArrayList<JmsSession> sessions = (inUseSessionsMap == null ? null : inUseSessionsMap.get(mode));
        if (sessions != null) {
            sessions.remove(session) ;
        }
        notifyAll() ;
    }
    
    /**
     * This method closes an open session without returning it to the pool.
     * @param session The session to be returned to the pool.
     * @throws SQLException
     */
    public synchronized void releaseSession(final JmsSession session) {
        session.handleReleaseSession(this) ;
    }
    
    /**
     * This method closes an open session without returning it to the pool.
     * @param session The session to be returned to the pool.
     * @throws SQLException
     * @deprecated
     */
    public synchronized void releaseSession(final Session session) {
      if (session instanceof JmsSession) {
          releaseSession((JmsSession)session) ;
      } else {
          logger.error("Invalid JMS Session type in releaseSession: " + session);
      }
    }

    /**
     * This method is called when the pool needs to destroyed. It closes all open sessions
     * and the connection and removes it from the container's poolMap.
     */
    public synchronized void removeSessionPool()
    {
        freeSessionsMap = null ;
        inUseSessionsMap = null ;
        transactionsToSessions = null ;
        sessionsToTransactions = null ;
        
        logger.debug("Emptied the session pool now closing the connection to the factory.");
        if (jmsConnection!=null) {
            try {
                jmsConnection.close();
            } catch (final Exception ex) {} // ignore
            jmsConnection=null;
            terminated = true ;
        }
        JmsConnectionPoolContainer.removePool(poolKey);
    }
    /**
     * Gets the total number of sessions in the pool regardless of the acknowlede mode
     * used when creating the sessions.
     * @return the session pool size
     */
    public int getSessionsInPool() {
        return getSessionsInPool(Session.AUTO_ACKNOWLEDGE) +
            getSessionsInPool(Session.CLIENT_ACKNOWLEDGE) +
            getSessionsInPool(Session.DUPS_OK_ACKNOWLEDGE) +
            getSessionsInPool(Session.SESSION_TRANSACTED) ;
    }
    
    /**
     * Returns the total nr of sessions for the specifed acknowledge mode
     * 
     * @param acknowledgeMode the acknowledge mode of sessions
     * @return
     */
    public synchronized int getSessionsInPool(final int acknowledgeMode) {
        return getFreeSessionsInPool(acknowledgeMode) + getInUseSessionsInPool(acknowledgeMode) ;
    }
    
    /**
     * Get the number of free sessions created with the specified acknowledge mode
     * @param acknowledgeMode the acknowledge mode of sessions
     * @return int	the number of in use sessions
     */
    public synchronized int getFreeSessionsInPool(final int acknowledgeMode) {
        final ArrayList<JmsSession> freeSessionMap = (freeSessionsMap == null ? null : freeSessionsMap.get(acknowledgeMode)) ;
        final int numFreeSessions = (freeSessionMap == null ? 0 : freeSessionMap.size()) ;
        return numFreeSessions;
    }
    
    /**
     * Get the number of sessions that are in use and that were
     * created with the specified acknowledge mode
     * @param acknowledgeMode the acknowledge mode of sessions
     * @return int	the number of in use sessions
     */
    public synchronized int getInUseSessionsInPool(final int acknowledgeMode) {
        final ArrayList<JmsSession> inUseSessionMap = (inUseSessionsMap == null ? null : inUseSessionsMap.get(acknowledgeMode)) ;
        final int numInUseSessions = (inUseSessionMap == null ? 0 : inUseSessionMap.size()) ;
        return numInUseSessions;
    }
    
    /**
     * Initialise the connection.
     * @throws ConnectionException If the pool has already been terminated.
     * @throws NamingContextException for errors obtaining a naming context
     * @throws NamingException for errors accessing a naming context
     * @throws JMSException for errors creating the connection
     */
    private synchronized void initConnection()
        throws ConnectionException, NamingContextException, NamingException, JMSException
    {
        if (terminated)
        {
            throw new ConnectionException("Connection pool has been terminated") ;
        }
        
        if (jmsConnection==null) {
            JmsConnectionPoolContainer.addToPool(poolKey, this);
            logger.debug("Creating a JMS Connection for poolKey : " + poolKey);
            Properties jndiEnvironment = JmsConnectionPoolContainer.getJndiEnvironment(poolKey);
            Context jndiContext = NamingContextPool.getNamingContext(jndiEnvironment);
            try {
                String connectionFactoryString = poolKey.get(JMSEpr.CONNECTION_FACTORY_TAG);
                Object factoryConnection=null;

                try
                {
                    factoryConnection = jndiContext.lookup(connectionFactoryString);
                } catch (NamingException ne) {
                    logger.info("Received NamingException, refreshing context.");
                    jndiContext = NamingContextPool.replaceNamingContext(jndiContext, JmsConnectionPoolContainer.getJndiEnvironment(poolKey));
                    factoryConnection = jndiContext.lookup(connectionFactoryString);
                }
                final String username = poolKey.get( JMSEpr.JMS_SECURITY_PRINCIPAL_TAG );
                final String password = poolKey.get( JMSEpr.JMS_SECURITY_CREDENTIAL_TAG );
                boolean useJMSSecurity = (username != null && password != null);
                logger.debug( "JMS Security principal [" + username + "] using JMS Security : " + useJMSSecurity );
                if (factoryConnection instanceof XAConnectionFactory) {
                    final XAConnectionFactory factory = (XAConnectionFactory)factoryConnection ;
                    jmsConnection = useJMSSecurity ? factory.createXAConnection(username,password): factory.createXAConnection();
                    isXAAware = true ;
                    freeSessionsMap.put(Session.SESSION_TRANSACTED, new ArrayList<JmsSession>() );
                    inUseSessionsMap.put(Session.SESSION_TRANSACTED, new ArrayList<JmsSession>() );
                } else if (factoryConnection instanceof ConnectionFactory) {
                    final ConnectionFactory factory = (ConnectionFactory)factoryConnection ;
                    jmsConnection = useJMSSecurity ? factory.createConnection(username,password): factory.createConnection();
                }
                
                jmsConnection.setExceptionListener(new ExceptionListener() {
                    public void onException(JMSException arg0)
                    {
                        removeSessionPool() ;
                    }
                }) ;
                jmsConnection.start();
            } finally {
                NamingContextPool.releaseNamingContext(jndiContext) ;
            }
        }
    }
    
    /**
     * Get the current transaction.
     * @return The transaction or null if none present.
     * @throws ConnectionException if the transaction context cannot be obtained.
     */
    private Object getTransaction()
        throws ConnectionException
    {
        try {
            return TransactionStrategy.getTransactionStrategy(true).getTransaction() ;
        } catch (final TransactionStrategyException tse) {
            throw new ConnectionException("Failed to determine current transaction context", tse) ;
        }
    }
    
    /**
     * Get a JMS session associated with the current transaction.
     * @return The JMS session or null if not associated. 
     * @throws ConnectionException For accessint the current transaction context
     */
    private synchronized JmsXASession getXASession()
        throws ConnectionException
    {
        final Object tx = getTransaction() ;
        return transactionsToSessions.get(tx) ;
    }
    
    /**
     * Associate the JMS XA Session with the current transaction.
     * @param session The XA session.
     * @throws ConnectionException if there is no transaction active.
     */
    synchronized void associateTransaction(final JmsXASession session)
        throws ConnectionException
    {
        final Object tx = getTransaction() ;
        if (tx == null)
        {
            throw new ConnectionException("No active transaction") ;
        }
        transactionsToSessions.put(tx, session) ;
        sessionsToTransactions.put(session, tx) ;
    }
    
    /**
     * Disassociate the JMS XA Session from a transaction.
     * @param session The XA session.
     */
    synchronized void disassociateTransaction(final JmsXASession session)
    {
        final Object tx = sessionsToTransactions.remove(session) ;
        transactionsToSessions.remove(tx) ;
    }
    
    static
    {
    	PropertyManager prop = ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE);
    	String value = prop.getProperty(Environment.JMS_CONNECTION_POOL_SIZE);
    	
    	if (value != null)
    	{
    		try
    		{
    			CONFIGURED_POOL_SIZE = Integer.parseInt(value);
    		}
    		catch (NumberFormatException ex)
    		{
    			ex.printStackTrace();
    		}
    	}
    	
    	value = prop.getProperty(Environment.JMS_SESSION_SLEEP);
    	
    	if (value != null)
    	{
    		try
    		{
    			CONFIGURED_SLEEP = Integer.parseInt(value);
    		}
    		catch (NumberFormatException ex)
    		{
    			ex.printStackTrace();
    		}
    	}
    }
}
