/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.couriers.helpers;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.SqlTableCourier;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.couriers.CourierServiceBindException;
import org.jboss.soa.esb.couriers.CourierTransportException;
import org.jboss.soa.esb.util.ClassUtil;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import java.net.URISyntaxException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.DriverManager;

/**
 * Factory for creating JDBCEpr based database resources for the SQLTableCourier..
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JDBCEprDBResourceFactory {

    private static Logger logger = Logger.getLogger(JDBCEprDBResourceFactory.class);

    private JDBCEpr epr;

    private DataSource dataSource;
    private String insertStatementSQL;
    private String listStatementSQL;
    private String select4UpdateStatementSQL;
    private String updateStatusStatementSQL;
    private String deleteStatementSQL;

    public JDBCEprDBResourceFactory(JDBCEpr epr) throws CourierServiceBindException {
        this.epr = epr;

        if (epr.getDatasource() != null) {
            lookupDataSource(epr);
        } else {
            try {
                try {
                    ClassUtil.forName(epr.getDriver(), getClass());
                } catch (ClassNotFoundException e) {
                    throw new CourierServiceBindException("Database driver '" + epr.getDriver() + "' not available on classpath.");
                }
            } catch (URISyntaxException e) {
                throw new RuntimeException("Unexpected URISyntaxException from EPR getter method.", e);
            }
        }
    }


    public Connection createConnection(boolean transactional) throws CourierTransportException, CourierServiceBindException {
        Connection connection;

        try {
            if(dataSource != null) {
                connection = dataSource.getConnection();
            } else {
                try {
                    connection = DriverManager.getConnection(epr.getURL(), epr.getUserName(), epr.getPassword());
                } catch (URISyntaxException e) {
                    throw new RuntimeException("Unexpected URISyntaxException from EPR getter method.", e);
                }
            }
        } catch (SQLException e) {
            throw new CourierTransportException("Failed to connect to DataSource.", e);
        }

        if (!transactional) {
            try {
                connection.setAutoCommit(false);
            } catch (SQLException e) {
                try {
                    connection.close();
                } catch (SQLException e1) {
                    logger.error("Failed to close connection.", e1);
                }
                throw new CourierTransportException("Failed to turn off autoCommit on connection..", e);
            }
        }

        return connection;
    }

    public PreparedStatement createListStatement(Connection connection) throws SQLException {
        if (listStatementSQL == null) {
            listStatementSQL = buildListStatementSQL();
        }
        return connection.prepareStatement(listStatementSQL);
    }

    public PreparedStatement createSelect4UpdateStatement(Connection connection) throws SQLException {
        if (select4UpdateStatementSQL == null) {
            select4UpdateStatementSQL = buildSelect4UpdateStatementSQL();
        }
        return connection.prepareStatement(select4UpdateStatementSQL);
    }

    public PreparedStatement createUpdateStatusStatement(Connection connection) throws SQLException {
        if (updateStatusStatementSQL == null) {
            updateStatusStatementSQL = buildUpdateStatusStatementSQL();
        }
        return connection.prepareStatement(updateStatusStatementSQL);
    }

    public PreparedStatement createInsertStatement(Connection connection) throws SQLException {
        if (insertStatementSQL == null) {
            insertStatementSQL = buildInsertStatementSQL();
        }
        return connection.prepareStatement(insertStatementSQL);
    }

    public PreparedStatement createDeleteStatement(Connection connection) throws SQLException {
        if (deleteStatementSQL == null) {
            deleteStatementSQL = buildDeleteStatementSQL();
        }
        return connection.prepareStatement(deleteStatementSQL);
    }

    private String buildSelect4UpdateStatementSQL() {
        StringBuilder sb = new StringBuilder("select ");

        try {
            if (!epr.getURL().contains("hsqldb")) {
                sb = sb.append(
                        epr.getDataColumn()).append(" from ").append(
                        epr.getTableName()).append(" where ").append(
                        epr.getMessageIdColumn()).append("=?").append(
                        " and ").append(epr.getStatusColumn())
                        .append("=?").append(" for update");
            } else {
                /*
                 * HSQL does not support FOR UPDATE! All tables appear to
                 * be inherently updatable!
                 */
                sb = sb.append(
                        epr.getDataColumn()).append(" from ").append(
                        epr.getTableName()).append(" where ").append(
                        epr.getMessageIdColumn()).append("=?").append(
                        " and ").append(epr.getStatusColumn())
                        .append("=?");
            }
        } catch (URISyntaxException e) {
            throw new RuntimeException("Unexpected URISyntaxException from EPR getter method.", e);
        }

        return sb.toString();
    }

    private String buildUpdateStatusStatementSQL() {
        try {
            StringBuilder sb = new StringBuilder("update ").append(
                    epr.getTableName()).append(" set ").append(
                    epr.getStatusColumn()).append("= ?").append(" where ")
                    .append(epr.getMessageIdColumn()).append("=?");

            return sb.toString();
        } catch (URISyntaxException e) {
            throw new RuntimeException("Unexpected URISyntaxException from EPR getter method.", e);
        }
    }

    private String buildInsertStatementSQL() {
        try {
            StringBuilder sb = new StringBuilder();

            sb.append("insert into ").append(epr.getTableName());
            sb.append(" (");
            sb.append(epr.getMessageIdColumn()).append(", ");
            sb.append(epr.getDataColumn()).append(", ");
            sb.append(epr.getStatusColumn()).append(", ");
            sb.append(epr.getTimestampColumn());
            sb.append(") values (?,?,?,?)");

            return sb.toString();
        } catch (URISyntaxException e) {
            throw new RuntimeException("Unexpected URISyntaxException from EPR getter method.", e);
        }
    }

    private String buildListStatementSQL() {
        StringBuilder sb = new StringBuilder();

        try {
            sb.append("select ");
            sb.append(epr.getMessageIdColumn()).append(", ");
            sb.append(epr.getTimestampColumn());
            sb.append(" from ").append(epr.getTableName());
            sb.append(" where ").append(epr.getStatusColumn());
            sb.append(" = '").append(SqlTableCourier.State.Pending.getColumnValue()).append("'");
            sb.append(" order by 2");
        } catch (URISyntaxException e) {
            throw new RuntimeException("Unexpected URISyntaxException from EPR getter method.", e);
        }

        return sb.toString();
    }

    private String buildDeleteStatementSQL() {
        try {
            StringBuilder sb = new StringBuilder("delete from ").append(
                    epr.getTableName()).append(" where ").append(
                    epr.getMessageIdColumn()).append(" =?");

            return sb.toString();
        } catch (URISyntaxException e) {
            throw new RuntimeException("Unexpected URISyntaxException from EPR getter method.", e);
        }
    }

    private void lookupDataSource(JDBCEpr epr) throws CourierServiceBindException {
        try
        {
            if (epr.getDatasource() != null) {
                InitialContext initContext;
                try {
                    initContext = new InitialContext();
                    dataSource = (DataSource) initContext.lookup(epr.getDatasource());
                } catch (NamingException e) {
                    logger.error("Problem resolving DataSource through JNDI", e);
                    throw e; // it'll get wrapped later anyway!
                }
            }
        }
        catch (final Exception e)
        {
            throw new CourierServiceBindException("Failed to lookup DataSource.", e);
        }
    }
}
